classdef class_REVS_sim_case < handle
    %class_REVS_sim_case
    %   holds input and output data sufficient to perform a single
    %   simulation
    
    properties
        name                    = '';   % typically the sim #
        enable                  = true;
        
        model                   = 'REVS_VM';
        sim_step_time           = 0.01; 
        sim_start_time_secs     = -5;   % sim start time [s]
        
        global_decimation       = 1;
        verbose                 = 0;
        
        logging_config class_REVS_logging_config;
        
        output_path             = 'output\';
        param_path              = '';
        script_path             = '';
        return_path             = '';
        sim_path                = '.';
        
        timestamp               = '';
        descriptor              = '';
        
        save_input_workspace    = false;
        save_output_workspace   = false;
        
        save_console            = false;
        show_console            = false;        
        console_log_file        = '';
        console_fid             = 1;
        
        sim_config struct       = struct();
        load_scripts cell       = {};
        preprocess_scripts cell = {};
        postprocess_scripts cell = {};
        
        %params                  = class_REVS_sim_params;
        %drive_cycle_fname;      % drive cycle filename or cell array of drive cycle filenames
        
        workspace struct       = struct(); % holds model workspace, including inputs and results
        
        batch_sim_ptr class_REVS_sim_batch;  % optional pointer to the batch_sim parent of this sim_case
    end
    
    methods
        
        function obj = class_REVS_sim_case()
            obj.logging_config          = class_REVS_logging_config;
            
            obj.return_path             = pwd;            
            obj.timestamp               = datestr(now,'yyyy_mm_dd_HH_MM');
            
        end
        
        
        function status = prepare_workspace(obj)
            
            start_dir = pwd;
            obj.prepare_console_log();
            
            cd(obj.param_path);
            
            % Load Configuration Structure
            REVS = class_REVS_setup();
            
            REVS.current_model = obj.model;
            
            REVS.sim_step_time_secs = obj.sim_step_time;
            REVS.sim_start_time_secs = obj.sim_start_time_secs;
            
            REVS.verbose = obj.verbose;
            REVS.global_decimation = obj.global_decimation;                        
            REVS.logging_config = obj.logging_config;
            
            REVS.output_fid = obj.console_fid;
                       
            sim_config = obj.sim_config;
            
                                               
            % Run preprocess scripts to modify the workspace
            if obj.run_workspace_scripts( obj.preprocess_scripts)               
                obj.log( ' --- SIMULATION PREP FAILURE ABORTING SIM CASE --- ');                  
                obj.close_console_log()
                status = 1;
                return
            end
            
            % Save the workspace
            varlist = who;  % get a list of workspace vars.
            varlist = setdiff(varlist,{'obj', 'start_dir'});
            for i = 1:length(varlist)
                obj.workspace.(varlist{i}) = eval(varlist{i});
            end
                        
            if obj.batch_sim_ptr.publish_params || obj.save_input_workspace
                % TODO:Fix this code for data stored in obj.workspace
                if obj.batch_sim_ptr.publish_params
                    publish_dir = [obj.output_path obj.descriptor '_publish\'];
                else
                    publish_dir = [obj.output_path obj.descriptor '_input\'];
                end
                
                mkdir(publish_dir);
                
                REVS.verbose = 3;   % default to full verbosity for published workspaces %TODO: See what this does
                
                if obj.batch_sim_ptr.publish_params
                    export2mfile([publish_dir obj.timestamp obj.descriptor '_' obj.name '_input_workspace.txt'],{'REVS','ambient','driver','drive_cycle','accessories','electric','controls','engine','transmission','vehicle'});  % publish simulation-ready workspace to plain text file
                end
                
                if obj.save_input_workspace || obj.batch_sim_ptr.publish_params
                    save([publish_dir obj.timestamp obj.descriptor '_' obj.name '_input_workspace'],'REVS','ambient','driver','drive_cycle','accessories','electric','controls','engine','transmission','vehicle');
                end
                
            end
            
            cd(start_dir);
            status = 0;
        
        end
            
        
        function status = sim(obj)
            
            start_dir = pwd;
            
            if ~obj.enable
                obj.log( ' --- Skipping Disabled Simulation --- ');
                status = -1;                                  
                return                
            end
            
            if isempty(fieldnames(obj.workspace))
                obj.log( 'Empty Workspace, Unable to Simulate');  
                obj.log( ' --- SIMULATION FAILURE ABORTING SIM CASE --- ');  
                status = 1;
                return
            end
            
            cd(obj.sim_path);

            % run simulation and pull outputs into current workspace
            warning('off','Simulink:blocks:BusSelectorOutSignalNameUpdate');

            obj.load_sim_workspace();

            sim_fail = [];
            if obj.console_fid > 1
                sim_console = evalc('try, simout = sim(obj.model,''SrcWorkspace'',''current'',''StopFcn'',''''); catch E, sim_fail = E; end');
                obj.log( '%s', sim_console);
            else
                try 
                    simout = sim(obj.model,'SrcWorkspace','current','StopFcn',''); 
                catch E 
                    sim_fail = E; 
                end
            end

            cd(start_dir);  % back to where we started

            if ~isempty( sim_fail )
                obj.log( 'Simulation Error: %s\n', sim_fail.message);  
                obj.log( ' --- SIMULATION FAILURE ABORTING SIM CASE --- ');  

                obj.close_console_log();
                status = 1;
                return
            end

            simout_fields = simout.get;
            for i = 1:length(simout_fields)
                eval([simout_fields{i} ' = simout.get(''' simout_fields{i} ''');']);
            end
            clear sim_console simout simout_fields;

             % run standard postprocessing
            [~, model_name] = fileparts(obj.model);
            eval(get_param(model_name, 'Stopfcn'))

            % Create starter model data
            model_data = class_test_data(datalog.time);

            % run logging based post-processing scripts, including model_data population
            % TODO: Figure how to wrap this in a try-catch while logging
            obj.log( '%s', evalc('REVS.logging_config.run_postprocess_scripts();'));

            if REVS.verbose && REVS.logging_config.has_package('REVS_log_fuel_economy')
                result.print( obj.console_fid);
            end

            audit.calc_audit();
            if REVS.verbose && REVS.logging_config.audit_type > 0
                audit.print(REVS.logging_config.audit_type, obj.console_fid);
            end

            if obj.run_workspace_scripts( obj.postprocess_scripts)               
                obj.log( ' --- SIMULATION POSTPROCESS FAILURE ABORTING SIM CASE --- ');                                                  
                status = 1;
            else
                status = 0;
            end

            % store output workspace
            temp.fids = who('*fid*','*_dir');
            temp.exclude_vars = {'ans','obj', temp.fids{:}, 'temp'};
            obj.workspace = export2struct('exclude', temp.exclude_vars);

            if obj.save_output_workspace
                temp.save_vars = setdiff(who, temp.exclude_vars);
                save([obj.output_path obj.timestamp obj.descriptor '_' obj.name '_output_workspace'], temp.save_vars{:});
            end
            
            % close console file
            obj.close_console_log()            
            

        end
        

        function status = run_workspace_scripts(obj, scripts)            
             
             for s = 1:numel(scripts)
                 
                 if obj.verbose
                     obj.log( 'Running script %s ...',  scripts{s});
                 end
                 
                 try
                     obj.log( '%s', evalc('evalin(''caller'',scripts{s})'));  
                 catch ex
                     rpt = regexprep( ex.getReport('extended','hyperlinks','off'), 'Error in class_REVS_sim_case/run_workspace_scripts .*', '');
                     obj.log( 'Case Script Error during %s: \n%s', scripts{s}, rpt);  
                     
                     
                     status = 1;
                     return
                 end
                  
             end
             status = 0;
         end
        
        function extract_workspace(obj)
            if isempty(obj.workspace)
                warndlg('Workspace not created or not retained');
                return
            end
                
            varlist = fieldnames(obj.workspace);
            for i = 1:length(varlist)
                assignin('caller', varlist{i}, obj.workspace.(varlist{i}));
            end
            evalin('caller','REVS.output_fid = 1;');    % clear possibly stale fid
        end
        
        function load_sim_workspace(obj)
            varlist = fieldnames(obj.workspace);
            for i = 1:length(varlist)
                assignin('caller', varlist{i}, obj.workspace.(varlist{i}));
            end
        end
               
        function store_sim_workspace(obj)
            varlist = evalin('caller','who');
            obj.workspace = struct();
          
            for v = 1:numel(varlist)
               obj.workspace.(varlist{v}) = evalin('caller', varlist{v});
            end            
        end
        
        function set.descriptor(obj, descriptor)
            descriptor = strrep(descriptor,' ','_');
                    
            if ~isempty(descriptor) && (descriptor(1) ~= '_')
               descriptor = ['_' descriptor];
            end
            
            obj.descriptor = descriptor;
        end
        
        
        function log(obj, varargin)
            
            msg = sprintf(varargin{:});
        
            if obj.save_console                 
                if obj.console_fid <= 1 
                    obj.console_fid = fopen(fullfile(obj.output_path,obj.console_log_file),'a');                    
                end
                fprintf( obj.console_fid, '%s\n', msg );                 
            end
        
            if obj.show_console
                fprintf( '%s\n', msg );  
            end
        end
                
        function prepare_console_log(obj)
            if obj.save_console && isempty( obj.console_log_file)
                obj.console_log_file = sprintf('%s%s_%s_console.txt', obj.timestamp, obj.descriptor , obj.name);
                obj.console_fid = fopen(fullfile(obj.output_path,obj.console_log_file),'w');
            end
        end
        
        function close_console_log(obj)
        
            if obj.console_fid > 1
                fclose(obj.console_fid);
            end            
            obj.console_fid = -1;
        end
            
        
    end
end

