classdef class_REVS_torque_converter
    %class_REVS_torque_converter
    %   Definition of class_REVS_torque_converter class
    
    properties
        
        pump_inertia_kgm2                               % Rotational Inertia of Torque Converter Input kgm^2
        turbine_inertia_kgm2                            % Rotational Inertia of Torque Converter Output kgm^2
        
        speed_ratio double {isvector}                    % Array of speed ratios
        drive_torque_ratio double {isvector}             % Torque multiplication in forward driving
        drive_k_factor_rpmpsqrtNm double {isvector}      % K factor in forward driving RPM/sqrt(Nm)
        
        decel_torque_ratio double {isvector}             % Torque multiplication in backward driving
        decel_k_factor_rpmpsqrtNm double {isvector}      % K factor in backward driving RPM/sqrt(Nm)
        
        lockup_duration_secs    = 0.3                   % Time for full torque converter clutch lockup
        lockup_efficiency_norm  = 1.00                  % Torque Transfer efficiency when locked up
        
        unlock_duration_secs	= 0.3;                  % Time for torque converter clutch to transition from full lockup to fully unlocked
        
        clutch_slip_Kp = 0.02;
        clutch_slip_Ki = 0.35;
        
    end
    
    properties ( Dependent = true, SetAccess = private, Transient = true)
        
        all_speed_ratios            % Combined Speed Ratios - Used for Simulation
        all_torque_ratios           % Combined Torque Ratios - USed for Simlation
        
        all_k_factors_rpmpsqrtNm    % Combined K Factors - Used for Simulation
        
    end
    
    
    methods
        
        %% Setters
        
        function obj = set.speed_ratio(obj,val)
            obj.speed_ratio = val(:);
        end
        
        function obj = set.drive_torque_ratio(obj,val)
            obj.drive_torque_ratio = val(:);
        end
        
        function obj = set.decel_torque_ratio(obj,val)
            obj.decel_torque_ratio = val(:);
        end
        
        function obj = set.drive_k_factor_rpmpsqrtNm(obj,val)
            obj.drive_k_factor_rpmpsqrtNm = val(:);
        end
        
        function obj = set.decel_k_factor_rpmpsqrtNm(obj,val)
            obj.decel_k_factor_rpmpsqrtNm = val(:);
        end
        
        
        
        %% Getters
        
        function val = get.all_speed_ratios(obj)
            val = [obj.speed_ratio(1:end-1); linspace(obj.speed_ratio(end),1.0,10)'];       % Extend to SR = 1
            val = [val; 1./val(end-1:-1:2)];                                                % Flip for Decel Portion
        end
        
        
        function val = get.all_torque_ratios(obj)
            combine_speed_ratio = [obj.speed_ratio;  1.0./obj.speed_ratio(end:-1:2) ];        
            combine_torque_ratio = [obj.drive_torque_ratio;  obj.decel_torque_ratio(end:-1:2)];
            val = interp1( combine_speed_ratio, combine_torque_ratio, obj.all_speed_ratios, 'linear',1 );
        end
        
        
        
        function val = get.all_k_factors_rpmpsqrtNm(obj)
            sr1_k_factor = 7.5 * max( obj.drive_k_factor_rpmpsqrtNm(end), obj.decel_k_factor_rpmpsqrtNm(end) );
            all_speed_ratios = obj.all_speed_ratios;
            drive_k_factors_rpmpsqrtNm   = interp1( [ obj.speed_ratio; 1.0], [ obj.drive_k_factor_rpmpsqrtNm; sr1_k_factor], all_speed_ratios(all_speed_ratios <=1) ,'spline');
            decel_k_factors_rpmpsqrtNm   = interp1( [ 1.0; 1.0./obj.speed_ratio(end:-1:2)], [sr1_k_factor; obj.decel_k_factor_rpmpsqrtNm(end:-1:2) ], all_speed_ratios(all_speed_ratios >=1)  ,'spline');
            val = [drive_k_factors_rpmpsqrtNm; decel_k_factors_rpmpsqrtNm(2:end)];
        end
        
        
    end
    
end

