classdef class_REVS_vehicle
    %class_REVS_vehicle
    %	Vehicle description class for REVS model for single axle drive
    
    properties       
        name;							% vehicle Description
        class;							% vehicle Type Specifier
        fuel;							% vehicle Fuel Specifier - Specified via class_REVS_fuel
        
        variant = 'default vehicle'                           % name of library vehicle variant target block
        powertrain_variant = 'CVM / P0';               % name of library powertrain variant target block
		driveline_variant = 'one axle drive';                 % name of library driveline variant target block
        controls_variant = 'CVM control';                      % name of library controls variant target block
        powertrain_type = enum_powertrain_type.conventional   % enumerated powertrain type
        
        delta_mass_static_kg class_REVS_dynamic_lookup;       % dynamic lookup for altering static mass during a drive cycle
        delta_mass_dynamic_kg class_REVS_dynamic_lookup;      % dynanic lookup for altering dynamic mass during a drive cycle
        
        use_abc_roadload logical;               % Use ABC roadload vs Cd & Crr
        
        coastdown_target_A_N;                   % coastdown target "A" term, SI units, Newtons
        coastdown_target_B_Npms;                % coastdown target "B" term, SI units, Newtons / (meter / second)
        coastdown_target_C_Npms2;               % coastdown target "C" term, SI units, Newtons / (meter / second)^2

        dyno_set_A_N;                           % dyno set "A" term, SI units, Newtons
        dyno_set_B_Npms;                        % dyno set "B" term, SI units, Newtons / (meter / second)
        dyno_set_C_Npms2;                       % dyno set "C" term, SI units, Newtons / (meter / second)^2
        
        coastdown_adjust_A_N = 0;               % coastdown adjustment for double counting "A" term, SI units, Newtons
        coastdown_adjust_B_Npms = 0;            % coastdown adjustment for double counting "B" term, SI units, Newtons / (meter / second)
        coastdown_adjust_C_Npms2 = 0;           % coastdown adjustment for double counting "C" term, SI units, Newtons / (meter / second)^2       
        
        frontal_area_m2;                        % vehicle frontal area, square meters
        aerodynamic_drag_coeff;                 % vehicle aerodynamic drag coefficient

        driveshaft class_REVS_driveshaft = class_REVS_driveshaft;    % Driveshaft Definition Class
        transfer_case class_REVS_transfer_case = class_REVS_transfer_case;   % Transfer Case Definition Class (For Class 8 vehicles, for example)

        steer_axle class_REVS_drive_axle = class_REVS_drive_axle;	% Drive Axle Definition Class
        drive_axle1 class_REVS_drive_axle = class_REVS_drive_axle;	% Drive Axle Definition Class
        drive_axle2 class_REVS_drive_axle = class_REVS_drive_axle;    % Drive Axle Definition Class (For Class 8 vehicles, for example)
        trailer_axle1 class_REVS_drive_axle = class_REVS_drive_axle;	% First Trailer Axle Definition Class (For Class 7 vehicles, for example)
        trailer_axle2 class_REVS_drive_axle = class_REVS_drive_axle;  % Second Trailer Axle Definition Class (For Class 8 vehicles, for example)
        
        max_brake_force_N;                      % Total combined braking torque of all axles        
	end
    
	properties (Hidden)
		
		set_mass_static_kg;						% Vehicle static mass (curb weight + ballast weight) ~ loaded test weight
        set_mass_dynamic_kg;					% vehicle equivalent inertia mass for simulation , e.g. static mass * 1.03
		set_mass_curb_kg;						% vehicle curb weight - unloaded, but full of fluids
		set_mass_ballast_kg;                    % vehicle ballast weight - mass added to get from curb weight to test weight
		
        powertrain_type_variant_map = containers.Map('KeyType', 'char', 'ValueType', 'char');        
	end
	
    properties ( Dependent )
        coastdown_target_A_lbf;                % coastdown target "A" term, SAE units, pounds force
        coastdown_target_B_lbfpmph;            % coastdown target "B" term, SAE units, pounds force / mph
        coastdown_target_C_lbfpmph2;           % coastdown target "C" term, SAE units, pounds force / mph^2

        dyno_set_A_lbf;                         % dyno set "A" term, SAE units, pounds force
        dyno_set_B_lbfpmph;                     % dyno set "B" term, SAE units, pounds force / mph
        dyno_set_C_lbfpmph2;                    % dyno set "C" term, SAE units, pounds force / mph^2

        coastdown_adjust_A_lbf;                % coastdown adjustment for double counting "A" term, SAE units, pounds force
        coastdown_adjust_B_lbfpmph;            % coastdown adjustment for double counting "B" term, SAE units, pounds force / mph
        coastdown_adjust_C_lbfpmph2;           % coastdown adjustment for double counting "C" term, SAE units, pounds force / mph^2
		
		ETW_kg;
		mass_static_kg;							% Vehicle static mass (curb weight + ballast weight) ~ loaded test weight
        mass_dynamic_kg;						% vehicle equivalent inertia mass for simulation , e.g. static mass * 1.03
		mass_curb_kg;							% vehicle curb weight - unloaded, but full of fluids
		mass_ballast_kg;                        % vehicle ballast weight - mass added to get from curb weight to test weight
		
		ETW_lbs;
		mass_curb_lbs;
		mass_ballast_lbs;
		mass_static_lbs;
		mass_dynamic_lbs;		
    end
    
    methods
        
        function obj = class_REVS_vehicle( powertrain_type )
	        
            obj.powertrain_type_variant_map(string(enum_powertrain_type.conventional))     = 'CVM / P0';
            obj.powertrain_type_variant_map(string(enum_powertrain_type.hybrid_P0))       = 'CVM / P0';
            obj.powertrain_type_variant_map(string(enum_powertrain_type.hybrid_P2))       = 'P2 Hybrid';
            obj.powertrain_type_variant_map(string(enum_powertrain_type.battery_electric)) = 'EVM';
            obj.powertrain_type_variant_map(string(enum_powertrain_type.no_powertrain))    = '';
            obj.powertrain_type_variant_map(string(enum_powertrain_type.hybrid_powersplit))    = 'Powersplit Hybrid';
            
            if nargin > 0
                obj.powertrain_type = powertrain_type;
%                 if (obj.powertrain_type == enum_powertrain_type.conventional)
                    obj.powertrain_variant = obj.powertrain_type_variant_map(string(obj.powertrain_type));
%                 end
            end
            
			obj.delta_mass_static_kg = class_REVS_dynamic_lookup;
            obj.delta_mass_static_kg.axis_1.signal       = 'cycle_phase_num';
            obj.delta_mass_static_kg.axis_1.breakpoints  = [0 1];
            obj.delta_mass_static_kg.table               = [0 0];
            
			obj.delta_mass_dynamic_kg = class_REVS_dynamic_lookup;
            obj.delta_mass_dynamic_kg.axis_1.signal      = 'cycle_phase_num';
            obj.delta_mass_dynamic_kg.axis_1.breakpoints = [0 1];
            obj.delta_mass_dynamic_kg.table              = [0 0];
            
            obj.use_abc_roadload = true; % default value for 99.99% of everything we've ever done
        end
        
        function roadload_power_kW = calc_roadload_power_kW(obj, speed_mps, use_adjusted , grade_pct)
        
            if nargin == 4                
                roadload_force_N = obj.calc_roadload_force_N(speed_mps, use_adjusted , grade_pct);
            elseif nargin == 3
                roadload_force_N = obj.calc_roadload_force_N(speed_mps, use_adjusted);                
            elseif nargin == 2
                roadload_force_N = obj.calc_roadload_force_N(speed_mps);
            else
                roadload_force_N = obj.calc_roadload_force_N();
                speed_mps = 50 * unit_convert.mph2mps;
            end
            
            roadload_power_kW = roadload_force_N .* speed_mps / 1000;
            
        end
        
        function roadload_force_N = calc_roadload_force_N(obj, speed_mps, use_adjusted , grade_pct)
                      
           if ~obj.use_abc_roadload               
               warning('Roadload calculated via CdA and Crr is an approximation');
               f0_N = obj.mass_static_kg .* obj.steer_axle.tire.vehicle_weight_norm .* obj.steer_axle.tire.rolling_resistance_coefficient;
               f0_N = f0_N + obj.mass_static_kg .* obj.drive_axle1.tire.vehicle_weight_norm .* obj.drive_axle1.tire.rolling_resistance_coefficient;
               f0_N = f0_N + obj.mass_static_kg .* obj.drive_axle2.tire.vehicle_weight_norm .* obj.drive_axle2.tire.rolling_resistance_coefficient;
               f0_N = f0_N + obj.mass_static_kg .* obj.trailer_axle1.tire.vehicle_weight_norm .* obj.trailer_axle1.tire.rolling_resistance_coefficient;
               f0_N = f0_N + obj.mass_static_kg .* obj.trailer_axle2.tire.vehicle_weight_norm .* obj.trailer_axle2.tire.rolling_resistance_coefficient;               
               
               f1_Npmps = 0;
               
               f2_Npmps2 = 0.5 *obj.aerodynamic_drag_coeff .* obj.frontal_area_m2 * 1.1677 ;               
           elseif nargin > 2 && use_adjusted           
                f0_N = obj.coastdown_target_A_N + obj.coastdown_adjust_A_N;
                f1_Npmps = obj.coastdown_target_B_Npms + obj.coastdown_adjust_B_Npms;
                f2_Npmps2 = obj.coastdown_target_C_Npms2 + obj.coastdown_adjust_C_Npms2;              
           else               
                f0_N = obj.coastdown_target_A_N;
                f1_Npmps = obj.coastdown_target_B_Npms;
                f2_Npmps2 = obj.coastdown_target_C_Npms2;               
           end
           
           
           if nargin < 2
              speed_mps = 50 * unit_convert.mph2mps; 
           end
           
           roadload_force_N = speed_mps .^ [2 1 0] * [f2_Npmps2 ; f1_Npmps ; f0_N];
           
           if nargin >= 4
              roadload_force_N = roadload_force_N + sin(atan(grade_pct/100)) * mass_kg * unit_convert.g;
           end
                       
           
                      
        end
        
        function val = get.max_brake_force_N(obj)
            val = REVS_class_default( obj.max_brake_force_N , obj.mass_static_kg * 9.8 * 1.75);  % changed from 2.0 to 1.4 for REVS1 compatible brake gain and alternator regen...
        end

        function val = get.steer_axle( obj )            
            %Set axle parameters that require vehicle level inputs
            val = obj.drive_axle1;
            
            val.brake.max_torque_Nm = REVS_class_default( val.brake.max_torque_Nm, obj.max_brake_force_N * val.brake.proportion_norm * val.tire.radius_m);
            
            val.tire.wheel_slip_torque_Nm = REVS_class_default( val.tire.wheel_slip_torque_Nm, obj.mass_static_kg * val.tire.vehicle_weight_norm * 9.81 * val.tire.radius_m);            
        end
                
        function val = get.drive_axle1( obj )
            %Set axle parameters that require vehicle level inputs
            val = obj.drive_axle1;
            
            val.brake.max_torque_Nm = REVS_class_default( val.brake.max_torque_Nm, obj.max_brake_force_N * val.brake.proportion_norm * val.tire.radius_m);
            
            val.tire.wheel_slip_torque_Nm = REVS_class_default( val.tire.wheel_slip_torque_Nm, obj.mass_static_kg * val.tire.vehicle_weight_norm * 9.81 * val.tire.radius_m);            
        end
        
        function val = get.drive_axle2( obj )
            %Set axle parameters that require vehicle level inputs
            val = obj.drive_axle1;
            
            val.brake.max_torque_Nm = REVS_class_default( val.brake.max_torque_Nm, obj.max_brake_force_N * val.brake.proportion_norm * val.tire.radius_m);
            
            val.tire.wheel_slip_torque_Nm = REVS_class_default( val.tire.wheel_slip_torque_Nm, obj.mass_static_kg * val.tire.vehicle_weight_norm * 9.81 * val.tire.radius_m);            
        end

        function val = get.trailer_axle1( obj )
            %Set axle parameters that require vehicle level inputs
            val = obj.trailer_axle1;
            
            val.brake.max_torque_Nm = REVS_class_default( val.brake.max_torque_Nm, obj.max_brake_force_N * val.brake.proportion_norm * val.tire.radius_m);
            
            val.tire.wheel_slip_torque_Nm = REVS_class_default( val.tire.wheel_slip_torque_Nm, obj.mass_static_kg * val.tire.vehicle_weight_norm * 9.81 * val.tire.radius_m);            
        end
        
        function val = get.trailer_axle2( obj )
            %Set axle parameters that require vehicle level inputs
            val = obj.trailer_axle2;
            
            val.brake.max_torque_Nm = REVS_class_default( val.brake.max_torque_Nm, obj.max_brake_force_N * val.brake.proportion_norm * val.tire.radius_m);
            
            val.tire.wheel_slip_torque_Nm = REVS_class_default( val.tire.wheel_slip_torque_Nm, obj.mass_static_kg * val.tire.vehicle_weight_norm * 9.81 * val.tire.radius_m);            
		end
        
		% setter / getters for dependent properties	- roadload coeficients
		
        function val = get.coastdown_target_A_N(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_target_A_N;
            else
                val = REVS_class_default(obj.coastdown_target_A_N,0);
            end
        end
        
        function val = get.coastdown_target_B_Npms(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_target_B_Npms;
            else
                val = REVS_class_default(obj.coastdown_target_B_Npms,0);
            end
        end
        
        function val = get.coastdown_target_C_Npms2(obj)
            if obj.use_abc_roadload
                val = obj.coastdown_target_C_Npms2;
            else
                val = REVS_class_default(obj.coastdown_target_C_Npms2,0);
            end
        end
                
        function val = get.dyno_set_A_N(obj)
            if obj.use_abc_roadload
                val = obj.dyno_set_A_N;
            else
                val = REVS_class_default(obj.dyno_set_A_N,0);
            end
        end
        
        function val = get.dyno_set_B_Npms(obj)
            if obj.use_abc_roadload
                val = obj.dyno_set_B_Npms;
            else
                val = REVS_class_default(obj.dyno_set_B_Npms,0);
            end
        end
        
        function val = get.dyno_set_C_Npms2(obj)
            if obj.use_abc_roadload
                val = obj.dyno_set_C_Npms2;
            else
                val = REVS_class_default(obj.dyno_set_C_Npms2,0);
            end
        end
        
        function val = get.aerodynamic_drag_coeff(obj)
            if obj.use_abc_roadload
                val = REVS_class_default(obj.aerodynamic_drag_coeff,0);
            else
                val = obj.aerodynamic_drag_coeff;
            end
        end
        
        function val = get.frontal_area_m2(obj)
            if obj.use_abc_roadload
                val = REVS_class_default(obj.frontal_area_m2,0);
            else
                val = obj.frontal_area_m2;
            end
        end
        
        function obj = calc_roadload_adjust(obj)
            % based on SAE 2020-01-1064 and the 2019 test car list
            % www.epa.gov/vehicle-and-fuel-emissions-testing/moskalik-using-transmission-data-isolate-individual-losses
            if (~isempty(obj.dyno_set_B_lbfpmph) && ~isempty(obj.dyno_set_C_lbfpmph2))
                obj.coastdown_adjust_B_lbfpmph  =   -99*(obj.dyno_set_C_lbfpmph2 - obj.coastdown_target_C_lbfpmph2);
                obj.coastdown_adjust_A_lbf      =   -22.9*(obj.coastdown_adjust_B_lbfpmph) - 5.0;              
                obj.coastdown_adjust_C_lbfpmph2 =   obj.dyno_set_C_lbfpmph2 - obj.coastdown_target_C_lbfpmph2;            
            else
                obj.coastdown_adjust_B_lbfpmph  =   - (0.796 * obj.coastdown_target_C_lbfpmph2 - 0.239);
                obj.coastdown_adjust_A_lbf      =   -22.9*(obj.coastdown_adjust_B_lbfpmph) - 5.0;              
                obj.coastdown_adjust_C_lbfpmph2 =   - obj.coastdown_adjust_B_lbfpmph / 99;            
            end
        end
        
		% setter / getters for dependent properties	- ETW & masses		
		function obj = set.ETW_kg( obj, val)
			obj.set_mass_static_kg = val * 0.985;
			obj.set_mass_dynamic_kg = val * 1.015; % 1.03 * .985 = 1.01455 ...
			obj.set_mass_ballast_kg = 300 * unit_convert.lbm2kg;
            obj.set_mass_curb_kg = obj.set_mass_static_kg - obj.set_mass_ballast_kg;
		end
		
		function val = get.ETW_kg( obj)
			val = obj.mass_static_kg / 0.985;
		end
				
		function val = get.mass_static_kg( obj) 
			if ~isempty( obj.set_mass_static_kg)
				val = obj.set_mass_static_kg;
			elseif ~isempty( obj.set_mass_dynamic_kg)
				val = obj.set_mass_dynamic_kg / 1.03;
			elseif ~isempty( obj.set_mass_curb_kg) %% && ~isempty( obj.set_mass_ballast_kg)
				val = obj.set_mass_curb_kg + obj.mass_ballast_kg;
			else 
				val = [];
			end
		end	
		
		function obj = set.mass_static_kg( obj, val)
			obj.set_mass_static_kg = val;
            obj.set_mass_dynamic_kg = obj.set_mass_static_kg / 0.985 * 1.015;
            obj.set_mass_curb_kg = obj.set_mass_static_kg - obj.set_mass_ballast_kg;
		end
		
		function val = get.mass_ballast_kg( obj) 
            if ~isempty( obj.set_mass_ballast_kg)
                val = obj.set_mass_ballast_kg;
            else
                val = 0;
            end
		end
		
		function obj = set.mass_ballast_kg( obj, val)
			obj.set_mass_ballast_kg = val;
		end
				
		function val = get.mass_curb_kg( obj)
			val = REVS_class_default(obj.set_mass_curb_kg, obj.mass_static_kg - obj.mass_ballast_kg);
		end
		
		function obj = set.mass_curb_kg( obj, val)
			obj.set_mass_curb_kg = val;
            obj.set_mass_static_kg = obj.set_mass_curb_kg + obj.set_mass_ballast_kg;
            obj.set_mass_dynamic_kg = obj.set_mass_static_kg / 0.985 * 1.015;
		end
		
		function val = get.mass_dynamic_kg( obj)
			val = REVS_class_default(obj.set_mass_dynamic_kg, obj.mass_static_kg * 1.03);
		end

		function obj = set.mass_dynamic_kg( obj, val)
			obj.set_mass_dynamic_kg = val;
		end
		
        % setter/getters for dependent properties unit conversion
        %% coastdown targets
        function obj = set.coastdown_target_A_lbf( obj, val )
            obj.coastdown_target_A_N = unit_convert.lbf2N * val ;
        end
        
        function val = get.coastdown_target_A_lbf( obj )
            val = unit_convert.N2lbf * obj.coastdown_target_A_N;
        end

        function obj = set.coastdown_target_B_lbfpmph( obj, val )
            obj.coastdown_target_B_Npms = unit_convert.lbf2N / unit_convert.mph2mps * val ;
        end
        
        function val = get.coastdown_target_B_lbfpmph( obj )
            val = unit_convert.N2lbf / unit_convert.mps2mph * obj.coastdown_target_B_Npms;
        end

        function obj = set.coastdown_target_C_lbfpmph2( obj, val )
            obj.coastdown_target_C_Npms2 = unit_convert.lbf2N / unit_convert.mph2mps / unit_convert.mph2mps * val ;
        end
        
        function val = get.coastdown_target_C_lbfpmph2( obj )
            val = unit_convert.N2lbf / unit_convert.mps2mph / unit_convert.mps2mph * obj.coastdown_target_C_Npms2;
        end

        %% dyno sets        
        function obj = set.dyno_set_A_lbf( obj, val )
            obj.dyno_set_A_N = unit_convert.lbf2N * val ;
        end
        
        function val = get.dyno_set_A_lbf( obj )
            val = unit_convert.N2lbf * obj.dyno_set_A_N;
        end

        function obj = set.dyno_set_B_lbfpmph( obj, val )
            obj.dyno_set_B_Npms = unit_convert.lbf2N / unit_convert.mph2mps * val ;
        end
        
        function val = get.dyno_set_B_lbfpmph( obj )
            val = unit_convert.N2lbf / unit_convert.mps2mph * obj.dyno_set_B_Npms;
        end

        function obj = set.dyno_set_C_lbfpmph2( obj, val )
            obj.dyno_set_C_Npms2 = unit_convert.lbf2N / unit_convert.mph2mps / unit_convert.mph2mps * val ;
        end
        
        function val = get.dyno_set_C_lbfpmph2( obj )
            val = unit_convert.N2lbf / unit_convert.mps2mph / unit_convert.mps2mph * obj.dyno_set_C_Npms2;
        end
        
        %% coastdown adjusts
        function obj = set.coastdown_adjust_A_lbf( obj, val )
            obj.coastdown_adjust_A_N = unit_convert.lbf2N * val ;
        end
        
        function val = get.coastdown_adjust_A_lbf( obj )
            val = unit_convert.N2lbf * obj.coastdown_adjust_A_N;
        end

        function obj = set.coastdown_adjust_B_lbfpmph( obj, val )
            obj.coastdown_adjust_B_Npms = unit_convert.lbf2N / unit_convert.mph2mps * val ;
        end
        
        function val = get.coastdown_adjust_B_lbfpmph( obj )
            val = unit_convert.N2lbf / unit_convert.mps2mph * obj.coastdown_adjust_B_Npms;
        end

        function obj = set.coastdown_adjust_C_lbfpmph2( obj, val )
            obj.coastdown_adjust_C_Npms2 = unit_convert.lbf2N / unit_convert.mph2mps / unit_convert.mph2mps * val ;
        end
        
        function val = get.coastdown_adjust_C_lbfpmph2( obj )
            val = unit_convert.N2lbf / unit_convert.mps2mph / unit_convert.mps2mph * obj.coastdown_adjust_C_Npms2;
        end
        
        %% mass variables
		function obj = set.ETW_lbs( obj, val )
            obj.ETW_kg = val * unit_convert.lbm2kg;
        end
        
        function val = get.ETW_lbs( obj )
            val = obj.ETW_kg * unit_convert.kg2lbm;
		end
		
		function obj = set.mass_curb_lbs( obj, val )
            obj.mass_curb_kg = val * unit_convert.lbm2kg;
        end
        
        function val = get.mass_curb_lbs( obj )
            val = obj.mass_curb_kg * unit_convert.kg2lbm;
		end
		
		function obj = set.mass_ballast_lbs( obj, val )
            obj.mass_ballast_kg = val * unit_convert.lbm2kg;
        end
        
        function val = get.mass_ballast_lbs( obj )
            val = obj.mass_ballast_kg * unit_convert.kg2lbm;
		end
		
		function obj = set.mass_static_lbs( obj, val )
            obj.mass_static_kg = val * unit_convert.lbm2kg;
        end
        
        function val = get.mass_static_lbs( obj )
            val = obj.mass_static_kg * unit_convert.kg2lbm;
		end
		
		function obj = set.mass_dynamic_lbs( obj, val )
            obj.mass_dynamic_kg = val * unit_convert.lbm2kg;
        end
        
        function val = get.mass_dynamic_lbs( obj )
            val = obj.mass_dynamic_kg * unit_convert.kg2lbm;
        end

        
    end
    
end

