function [engine] = REVS_build_engine( engine_base,  core_data, varargin )
%REVS_build_engine Build REVS engine from source data
%
% Inputs:    engine,  map, WOT, CT, varargin
%
% engine - Skeleton engine (structure or class), must define displacement
%
% map - structure containing fuel map test data.  Must contain fields from
%		which speed torque and fuel can be obtained.  Available field names:
%		speed_radps
%		speed_rpm
%		torque_Nm
%		bmep_bar
%		fuel_gps
%		bsfc_gpkWhr
%		
%
%
% WOT - optional structure defining maximum torque curve points or test data.
%		Naming scheme same as map.  If omitted use empty matrix [], and
%		convex hull of map data will be used to approximate curve.
%
% CT -  optional structure defining minimum torque curve points or test data.
%		Naming scheme same as map.  If omitted use empty matrix [], and
%		convex hull of map data will be used to approximate curve. If no fuel
%		map data is avaiable below 0 torque a default curve is scaled based on
%		engine displacement.
%
% Varargin Options
%
%

validate_arg( engine_base, {'class_REVS_engine','class_REVS_engine_legacy','struct'} );
validate_arg( core_data, {'cell','struct'} );


% Separate WOT test data - torque & speed Only
WOT_data = parse_varargs( varargin, 'WOT', [], {'cell','struct'});

% Separate closed throtte data - torque & speed only
CT_data = parse_varargs( varargin, 'CT', [], {'cell','struct'});

% Cylinder Deac Data
deac_data = parse_varargs( varargin,'deac',[]);
deac_expand_factor = parse_varargs( varargin,'deac_expansion',1.1,'numeric');


class_type = parse_varargs(varargin,'class','class_REVS_engine','char');

image_input = parse_varargs(varargin,'image',false,'toggle');

% show_plots = parse_varargs(varargin,'no_plots',true,'toggle');

plot_voronoi = parse_varargs(varargin,'plot_voronoi',false,'toggle');
plot_grid = parse_varargs(varargin,'plot_grid',false,'toggle');

plot_bsfc = parse_varargs(varargin,'plot_bsfc',false,'toggle');
plot_efficiency = parse_varargs(varargin,'plot_efficiency',false,'toggle');
plot_fuel_per_injection = parse_varargs(varargin,'plot_fuel_per_injection',false,'toggle');

plot_WOT_data = parse_varargs(varargin,'no_WOT_data',true,'toggle');
plot_CT_data = parse_varargs(varargin,'no_CT_data',true,'toggle');

show_point_labels = parse_varargs(varargin,'no_point_labels',true,'toggle');

torque_alpha = parse_varargs(varargin,'torque_alpha',0.1,'numeric',{'scalar'});

% Gridfit smoothing
smooth = parse_varargs(varargin,'smoothing',0.0008,'numeric',{'scalar'});
y_smooth = parse_varargs(varargin,'y_smoothing',smooth,'numeric',{'scalar'});
x_smooth = parse_varargs(varargin,'x_smoothing',(2.0 - image_input*0.9)*smooth,'numeric',{'scalar'});

grid_speed_radps = parse_varargs( varargin,'map_speed',[],'numeric',{'vector'});
grid_torque_Nm = parse_varargs( varargin,'map_torque',[],'numeric',{'vector'});

CT_torque_est_method = parse_varargs(varargin,'CT_est_method','enable_zero_fuel_corr_CT','char');

%% Construct Class to fill
% Copy provided info into engine class
engine = eval( class_type );

if ~isfield(engine_base,'idle_target_speed_radps') || isa(engine_base.idle_target_speed_radps,'class_REVS_dynamic_lookup')
    % No need to modify idle
elseif isnumeric(engine_base.idle_target_speed_radps) && isscalar( engine_base.idle_target_speed_radps)
	temp = engine_base.idle_target_speed_radps;
	engine_base.idle_target_speed_radps = class_REVS_dynamic_lookup;
	engine_base.idle_target_speed_radps.axis_1.signal = 'eng_runtime_sec';
	engine_base.idle_target_speed_radps.axis_1.breakpoints = [0,10];
	engine_base.idle_target_speed_radps.table = temp * [1,1];
else
	error('Unknown input for idle_speed_radps, must be a class_REVS_dynamic_lookup or scalar numeric value')
end

fields = fieldnames(engine_base);
for f = 1:length(fields)
	engine.(fields{f}) = engine_base.(fields{f});
end



%% Generate any missing fields in the provided input_data
% data needs  speed in radps, torque in Nm, fuel in g/sec

core_data = convert_and_combine( core_data, engine.displacement_L, 1);
WOT_data = convert_and_combine( WOT_data, engine.displacement_L, 0);
CT_data = convert_and_combine( CT_data, engine.displacement_L, 0);
deac_data = convert_and_combine( deac_data, engine.displacement_L, 1);

% Set tolerance for generating WOT and CT curves
if ~isempty(WOT_data)
	tolerance = max(WOT_data.torque_Nm) * 0.01;
else
	tolerance = max( core_data.torque_Nm) * 0.01;
end

%% Handle WOT Curve
[speed_radps, torque_Nm] = process_full_throttle( WOT_data, core_data, tolerance, torque_alpha);

% Define Wide-Open Throttle (WOT) Torque Table
engine.full_throttle_speed_radps  = speed_radps;
engine.full_throttle_torque_Nm = torque_Nm;

%% Handle CTP Curve
[speed_radps, torque_Nm] = process_closed_throttle( CT_data, core_data, max(engine.full_throttle_speed_radps), engine.displacement_L,  tolerance, torque_alpha);

% Define Closed Throttle Torque Table
engine.closed_throttle_speed_radps     = speed_radps;
engine.closed_throttle_torque_Nm       = torque_Nm;

%% Construct Output Map Grid
if ~isempty( grid_speed_radps )
	% Use Provided Points
elseif image_input
	max_rpm = ceil(engine.full_throttle_speed_radps(end)* unit_convert.radps2rpm/500)*500;
	grid_speed_radps = unique([0, 500:250:max_rpm/2, max_rpm:-500:max_rpm/2]) * unit_convert.rpm2radps;
else
	grid_speed_radps = find_clusters( core_data.speed_radps);
	grid_speed_radps(end+1) = min(grid_speed_radps)*0.5;
	grid_speed_radps(end+1) = 0;
	grid_speed_radps(end+1) = max(engine.full_throttle_speed_radps);
	grid_speed_radps = fill_gaps(grid_speed_radps, max(grid_speed_radps)/12);
end

if ~isempty(grid_torque_Nm)
	% Use Provided Points
else
	
	if image_input
		grid_torque_Nm = linspace(  min(core_data.torque_Nm),  max(engine.full_throttle_torque_Nm), 25);
	else
		grid_torque_Nm = find_clusters( core_data.torque_Nm);	
	end
	
	grid_torque_Nm(end+1) = 1.05* min(engine.closed_throttle_torque_Nm);
	grid_torque_Nm(end+1) = 1.05* max(engine.full_throttle_torque_Nm);
	grid_torque_Nm = fill_gaps(grid_torque_Nm, max(grid_torque_Nm)/15);
end	
	
%% Handle Naturally Aspirated Torque Curve

if ~all(isnan( core_data.manifold_press_kPaA))
	%Clean out NaNs
	clean_MAP_data = ~isnan( core_data.manifold_press_kPaA);
	MAP_data.speed_radps = core_data.speed_radps(clean_MAP_data);
	MAP_data.torque_Nm = core_data.torque_Nm(clean_MAP_data);
	MAP_data.pressure_kPa = core_data.manifold_press_kPaA(clean_MAP_data);
		
	temp =  scatter2surf(MAP_data.speed_radps,  MAP_data.pressure_kPa, MAP_data.torque_Nm, grid_speed_radps,[0,100,200],'xnormalize','ynormalize','method','gridfit','xsmooth',0.001,'ysmooth',0.001); % ,'contour'
		
	[na_speed_radps,na_torque_Nm] = build_curve(grid_speed_radps, temp( 2,:), tolerance);
	
	engine.naturally_aspirated_speed_radps = na_speed_radps;
	engine.naturally_aspirated_torque_Nm = na_torque_Nm;
	
elseif max(engine.full_throttle_torque_Nm) >  15 / ( 4 * pi / (engine.displacement_L /1000)*1e-5 )
	% Assume Turbo if > 15 bar BMEP - set NA torque @ 9 bar BMEP
	engine.naturally_aspirated_speed_radps = engine.full_throttle_speed_radps([1,end]);
	engine.naturally_aspirated_torque_Nm = 9 / ( 4 * pi / (engine.displacement_L /1000)*1e-5 ) * [1,1];
else
	% Assume NA
	engine.naturally_aspirated_speed_radps = [];
	engine.naturally_aspirated_torque_Nm = [];
end

%% Construct Fuel Map

% Select clean fuel data
clean_fuel_idx = ~isnan( core_data.fuel_gps);

fuel_data.speed_radps = core_data.speed_radps(clean_fuel_idx);
fuel_data.torque_Nm = core_data.torque_Nm(clean_fuel_idx);
fuel_data.fuel_gps = core_data.fuel_gps(clean_fuel_idx);


% interpolate map scatter points and grid to generate a surface matrix
speed_scale = engine.max_power_avg_speed_radps;
torque_scale = engine.max_torque_Nm;


map_fit_fuel_gps = scatter2surf(fuel_data.speed_radps, fuel_data.torque_Nm, fuel_data.fuel_gps , grid_speed_radps, grid_torque_Nm,'method','gridfit','interp','bilinear','xscale',speed_scale,'yscale',torque_scale,'xsmooth',x_smooth,'ysmooth',y_smooth,'spatial_weighting', 0.95);

% Meshgrid for output
[fuel_mesh_speed_radps, fuel_mesh_torque_Nm] = meshgrid( grid_speed_radps, grid_torque_Nm );
	
% Limit min BSFC to match data
data_min_bsfc_gpkWhr = min(core_data.bsfc_gpkWhr( core_data.torque_Nm > 0 & core_data.speed_radps > 10 )); 
map_min_fuel_gps = ( data_min_bsfc_gpkWhr /1000 /3600 ) .* (fuel_mesh_speed_radps .* fuel_mesh_torque_Nm );

adjust_points = map_fit_fuel_gps < map_min_fuel_gps;

map_out_fuel_gps = map_fit_fuel_gps;
map_out_fuel_gps(adjust_points) = 0.30 * map_fit_fuel_gps(adjust_points) + 0.70 * map_min_fuel_gps(adjust_points);
% map_out_fuel_gps(map_out_fuel_gps<0) = 0.01;

% assign the grid vectors to the surface X and Y definitions.
engine.fuel_map_speed_radps = grid_speed_radps;
engine.fuel_map_torque_Nm = grid_torque_Nm;
% engine.fuel_map_gps = max(map_out_fuel_gps, 0); % change negative values to 0
engine.fuel_map_gps = map_out_fuel_gps; % Used this condition to prevent non-linearity observed in the fuel plot with the previous condition


interp_fuel = interp2( grid_speed_radps, grid_torque_Nm, engine.fuel_map_gps, fuel_data.speed_radps, fuel_data.torque_Nm);
residual = interp_fuel - fuel_data.fuel_gps;
residual_pct = residual ./ fuel_data.fuel_gps;


%% Adjust CTP curve

if isempty(CT_data) && sum( fuel_data.torque_Nm < 0 ) < 5
[speed_radps, torque_Nm] = adjust_closed_throttle(engine.closed_throttle_speed_radps, engine.closed_throttle_torque_Nm,  grid_speed_radps, grid_torque_Nm, map_out_fuel_gps, max(engine.full_throttle_speed_radps), tolerance, CT_torque_est_method);

% Define Closed Throttle Torque Table
engine.closed_throttle_speed_radps     = speed_radps;
engine.closed_throttle_torque_Nm       = torque_Nm;
end

%% Handle Cylinder Deac

if ~isempty(deac_data)
	
	% Fuel map for cylinder deac [g/sec]
	deac_out_fuel_gps = scatter2surf(deac_data.speed_radps, deac_data.torque_Nm, deac_data.fuel_gps, grid_speed_radps, grid_torque_Nm,'method','gridfit','interp','bilinear','xscale',speed_scale,'yscale',torque_scale,'xsmooth',x_smooth,'ysmooth',y_smooth);
	
	% Always Less than base map
	engine.deac_fuel_map_gps = max(min(deac_out_fuel_gps,engine.fuel_map_gps), 0.0);
	
	% Compute deac region
	k = convhull(deac_data.speed_radps, deac_data.torque_Nm);
	deac_hull_speed_radps = deac_data.speed_radps(k);
	deac_hull_torque_Nm = deac_data.torque_Nm(k);
	next_pt = [2:length(k), 1];
	
	% Compute Centroid
	deac_cent_speed_radps = sum( (deac_hull_speed_radps + deac_hull_speed_radps(next_pt)) .* ( deac_hull_speed_radps  .* deac_hull_torque_Nm(next_pt) - deac_hull_speed_radps(next_pt) .* deac_hull_torque_Nm) )/6  /( sum( deac_hull_speed_radps  .* deac_hull_torque_Nm(next_pt) - deac_hull_speed_radps(next_pt) .* deac_hull_torque_Nm)/2);
	deac_cent_torque_Nm = sum( (deac_hull_torque_Nm + deac_hull_torque_Nm(next_pt)) .* ( deac_hull_speed_radps  .* deac_hull_torque_Nm(next_pt) - deac_hull_speed_radps(next_pt) .* deac_hull_torque_Nm) )/ 6 / ( sum( deac_hull_speed_radps  .* deac_hull_torque_Nm(next_pt) - deac_hull_speed_radps(next_pt) .* deac_hull_torque_Nm)/2);
	
	% Expand by 30%
	deac_expand_speed_radps = deac_cent_speed_radps + deac_expand_factor * (deac_hull_speed_radps - deac_cent_speed_radps);
	deac_expand_torque_Nm = deac_cent_torque_Nm + deac_expand_factor * (deac_hull_torque_Nm - deac_cent_torque_Nm);
		
	% Mask Outside excess extrapolation
	deac_mask = inpolygon( fuel_mesh_speed_radps, fuel_mesh_torque_Nm, deac_expand_speed_radps, deac_expand_torque_Nm);
	engine.deac_fuel_map_gps(~deac_mask) = nan;
	
end


%% Plot Results for Examination

% Voronoi Diagram
if plot_voronoi
	figure;
	[vx, vy] = voronoi(core_data.speed_radps/speed_scale,core_data.torque_Nm/torque_scale);
	superplot(core_data.speed_radps * unit_convert.radps2rpm,core_data.torque_Nm,'r+7');
	ax = axis;
	hold on
	plot(vx*speed_scale* unit_convert.radps2rpm,vy*torque_scale,'b-');
	axis(ax);
	title( 'Voronoi Diagram');
	xlabel('Speed (RPM)');
	ylabel('Torque (Nm)');
end

% Grid Selection
if plot_grid
	
	% Show Grid
	REVS_plot_engine(engine,'fuel flow', 'deac_prop', 0, 'show_bmep','show_CTP_curve','no_min_bsfc','no_power_lines','no_na_curve');
	hold on
	
	for i = 1:length(engine.fuel_map_speed_radps)
		superplot(unit_convert.radps2rpm * 	engine.fuel_map_speed_radps(i),[-intmax,intmax],'gy-1');
	end
	
	for i = 1:length(engine.fuel_map_torque_Nm)
		superplot([-intmax,intmax],engine.fuel_map_torque_Nm(i),'gy-1');
	end
	
	superplot( core_data.speed_radps * unit_convert.radps2rpm, core_data.torque_Nm,'rro4');
	for i = 1:length(core_data.fuel_gps)
% 		text( core_data.speed_radps(i) * unit_convert.radps2rpm+ 35, core_data.torque_Nm(i), num2str(core_data.fuel_gps(i),'%03.2f' ));
	end
	
end

% Contour of BSFC, Efficiency, etc. 
if plot_bsfc
	plot_contour( engine, core_data, false,'BSFC', WOT_data, plot_WOT_data, CT_data, plot_CT_data, show_point_labels)
	
	if ~isempty(deac_data)
		plot_contour( engine, deac_data, true,'BSFC', WOT_data, plot_WOT_data, CT_data, plot_CT_data, show_point_labels)
		superplot( deac_hull_speed_radps * unit_convert.radps2rpm , deac_hull_torque_Nm, 'gy--1');
		superplot( deac_expand_speed_radps * unit_convert.radps2rpm , deac_expand_torque_Nm, 'gy--1');
	end
	
end

if plot_efficiency
	plot_contour( engine, core_data, false,'Efficiency', WOT_data, plot_WOT_data, CT_data, plot_CT_data, show_point_labels)
	
	if ~isempty(deac_data)
		plot_contour( engine, deac_data, true,'Efficiency', WOT_data, plot_WOT_data, CT_data, plot_CT_data, show_point_labels)
		superplot( deac_hull_speed_radps * unit_convert.radps2rpm , deac_hull_torque_Nm, 'gy--1');
		superplot( deac_expand_speed_radps * unit_convert.radps2rpm , deac_expand_torque_Nm, 'gy--1');
	end
			
end

if plot_fuel_per_injection
	plot_contour( engine, core_data, false, 'FUEL PER INjection', WOT_data, plot_WOT_data, CT_data, plot_CT_data, show_point_labels)
		
	if ~isempty(deac_data)
		plot_contour( engine, deac_data, true,'FUEL PER INjection', WOT_data, plot_WOT_data, CT_data, plot_CT_data, show_point_labels)
		superplot( deac_hull_speed_radps * unit_convert.radps2rpm , deac_hull_torque_Nm, 'gy--1');
		superplot( deac_expand_speed_radps * unit_convert.radps2rpm , deac_expand_torque_Nm, 'gy--1');
	end

end


end

function plot_contour( engine, core_data, is_deac, data_select, WOT_data, plot_WOT_data, CT_data, plot_CT_data, show_point_labels)

	addtl_opts = {};
	
	switch upper(data_select)
		case 'EFFICIENCY'
			data_points =  100 * ((core_data.speed_radps .* core_data.torque_Nm) ./ 1000 ) ./ ( core_data.fuel_gps .* engine.fuel.energy_density_MJpkg );
		case 'BSFC'
			data_points = (core_data.fuel_gps .* 3600 .* 1000) ./ max(eps,core_data.speed_radps .* core_data.torque_Nm);		
		case 'FUEL PER INJECTION'
			data_points =  core_data.fuel_gps *60 ./ (core_data.speed_radps * unit_convert.radps2rpm ) / engine.num_cylinders * 2*1000;
	end
	
	% Find valid BSFC for contours
	valid_idx = ~isnan(core_data.bsfc_gpkWhr) & ~isinf(core_data.bsfc_gpkWhr) & core_data.bsfc_gpkWhr > 0 ;
	valid_points = data_points(valid_idx);
	
	round_points= 0.1 * round(valid_points*10);
	unique_points = unique(round_points);
	
	level_counts = histcounts( round_points, unique_points);
	keep = level_counts > max(2,0.01 * length(valid_points));
		
	%Custom countours to match data?
	if sum(level_counts(keep)) > 0.9 * length(valid_points) 
					
		contour_levels =unique_points(keep);
		
		switch upper(data_select)
			case 'EFFICIENCY'
				if contour_levels(1) > 7
					contour_levels = [(5:5:(contour_levels(1)-2))'; contour_levels ];
				end
			case 'BSFC'
				if contour_levels(end) < 280
					contour_levels = [contour_levels;300;400;500;1000];
				elseif contour_levels(end) < 350
					contour_levels = [contour_levels;400;500;1000];
				elseif contour_levels(end) < 400
					contour_levels = [contour_levels;500;1000];
				elseif contour_levels(end) < 800
					contour_levels = [contour_levels;1000];
				end
				
		end
		

		addtl_opts{end+1}=  'contours';
		addtl_opts{end+1}=  contour_levels;
		
	end
	
	% Draw efficiency plot
	if is_deac
		REVS_plot_engine(engine,data_select,'show_CTP_curve','no_min_bsfc','show_bmep','deac_prop','deac',addtl_opts{:});
	else
		REVS_plot_engine(engine,data_select,'show_CTP_curve','no_min_bsfc','show_bmep',addtl_opts{:});
	end
	
	% Draw WOT
	if ~isempty(WOT_data) && plot_WOT_data
		max_torque_data_hand = superplot( WOT_data.speed_radps * unit_convert.radps2rpm, WOT_data.torque_Nm,1.0,'ororo2');
	else
		max_torque_data_hand = [];
	end
	max_torque_out_hand = superplot( engine.full_throttle_speed_radps * unit_convert.radps2rpm, engine.full_throttle_torque_Nm ,1.0, 'bo9');
	
	% Draw CT
	if ~isempty(CT_data) && plot_CT_data
		min_torque_data_hand = superplot( CT_data.speed_radps * unit_convert.radps2rpm, CT_data.torque_Nm,1.0, 'ororo2');
	else
		min_torque_data_hand = [];
	end
	
	min_torque_out_hand = superplot( engine.closed_throttle_speed_radps * unit_convert.radps2rpm, engine.closed_throttle_torque_Nm ,1.0, 'bo9');
	
	% Draw Data points
	datasets = unique(core_data.dataset)';
	data_colors = {'k','lg','m','db','dg','c'};
	line_handles = [];
	
	for d = datasets
		pts = core_data.dataset == d;
		
		if length( core_data.speed_radps) > 500
			line_handles(end+1) = superplot( core_data.speed_radps(pts) * unit_convert.radps2rpm, core_data.torque_Nm(pts), [data_colors{d},data_colors{d},'o2']);
		else
			line_handles(end+1) = superplot( core_data.speed_radps(pts) * unit_convert.radps2rpm, core_data.torque_Nm(pts), [data_colors{d},data_colors{d},'o3']);
		end
		
		if show_point_labels
			
			if ismember( upper(data_select), { 'EFFICIENCY','BSFC'})
				pts = pts & core_data.torque_Nm > 2;
			end			
			h = text( core_data.speed_radps(pts) * unit_convert.radps2rpm, core_data.torque_Nm(pts), num2str(data_points(pts),'%02.1f' ),'FontSize',8,'VerticalAlignment','bottom');
			set(h,'Clipping','on');
		end
	end
	
	% Legend
	
	legend_text = core_data.names;
     
    if ~isempty( max_torque_data_hand) && ~isempty(min_torque_data_hand)
        line_handles(end+1) = max_torque_data_hand;
        legend_text{end+1} = 'Min & Max Torque Data';
    elseif ~isempty( max_torque_data_hand)
        line_handles(end+1) = max_torque_data_hand;
        legend_text{end+1} = 'Max Torque Data';
    elseif ~isempty( min_torque_data_hand)
        line_handles(end+1) = min_torque_data_hand;
        legend_text{end+1} = 'Min Torque Data';
    elseif ~isempty( min_torque_data_hand)
        line_handles(end+1) = min_torque_data_hand;
        legend_text{end+1} = 'Min Torque Data';
    end
    
	
	% Add 10% more space above for legend
	ylim(ylim .* [1,1.05]);
	
	line_handles(end+1) = max_torque_out_hand;
	legend_text{end+1} = 'Selected Min & Max Torque Points';

	leg = legend( line_handles, legend_text ,'Location','Best');%,'Orientation','Horizontal');%,'box','off');
	leg_pos = get(leg,'position');
	
	leg_pos(1) = 0.98 - leg_pos(3);
	leg_pos(2) = 0.93 - leg_pos(4);
	
	set(leg,'position',leg_pos);
	
end


function [speed_radps, torque_Nm] = process_full_throttle( WOT, map, tol, user_alpha)

if ~isempty(WOT) && length(WOT.speed_radps ) > 10
	% > 50 points => Test Data need to reduce
	
	% Add point at origin
	speed_radps = [0; WOT.speed_radps(:)];
	torque_Nm = [0; WOT.torque_Nm(:)];

	
	% Filter provided data by averaging into bins
	[speed_radps,torque_Nm] =  gen_avg_curve(speed_radps, torque_Nm, 100 * unit_convert.rpm2radps);

else
	%Combine map data with any points provided
	if ~isempty( WOT)
		speed_pts_radps = [map.speed_radps;WOT.speed_radps;min(map.speed_radps); max(map.speed_radps)];
		torque_pts_Nm = [map.torque_Nm;WOT.torque_Nm;-eps;-eps];
	else
		speed_pts_radps = [map.speed_radps;min(map.speed_radps); max(map.speed_radps)];
		torque_pts_Nm = [map.torque_Nm;-eps;-eps];
	end
	
	power_kW = speed_pts_radps .* torque_pts_Nm / 1000;
	[max_power_kW, max_power_idx] = max(power_kW);
	speed_scale = speed_pts_radps(max_power_idx);
	torque_scale = max(torque_pts_Nm);
	
	% Scale and keep Unique
	speed_pts_norm = speed_pts_radps / speed_scale;
	torque_pts_norm = torque_pts_Nm / torque_scale;

	% Fill down and left (0.01) and right(0.2) adding points
	speed_pts_norm = speed_pts_norm +  0.2*( 0:0.1:1);
	torque_pts_norm = torque_pts_norm - (0:0.1:1);
	speed_pts_norm = speed_pts_norm(:);
	torque_pts_norm = torque_pts_norm(:);
	
	% Keep just uniques
	[unique_pts_norm, unique_in_idx] = unique( [speed_pts_norm , torque_pts_norm],'rows');

	
	%Compute ALpha Shape
	shp = alphaShape(unique_pts_norm);
	crit_alpha = shp.criticalAlpha('one-region');
	shp.Alpha = max( crit_alpha, user_alpha);

	bound_idx  = shp.boundaryFacets();
	bound_idx = bound_idx([end:-1:1],1); % Flip for clockwise

	speed_radps = speed_scale * speed_pts_norm(unique_in_idx(bound_idx));
	torque_Nm = torque_scale * torque_pts_norm(unique_in_idx(bound_idx));

	% Shift start point to minimum speed
	start_idx = find( torque_Nm < 0, 1);
	speed_radps = circshift( speed_radps, -(start_idx-1) );
	torque_Nm = circshift( torque_Nm, -(start_idx-1) );
	
	% Remove added high speed points
	remove = speed_radps > speed_scale;
	speed_radps(remove) = [];
	torque_Nm(remove) = [];
	
	% Remove negative torque points
	remove = torque_Nm <= 0;
	speed_radps(remove) = [];
	torque_Nm(remove) = [];
	
	% Remove endpoints of vertical segments
	remove = [( abs(diff( speed_radps)) <= 2.5e-3*speed_scale & diff( torque_Nm) > 0) ; false] | [ false ; (abs(diff( speed_radps)) <= 1e-3*speed_scale & diff( torque_Nm) < 0) ];
	speed_radps(remove) = [];
	torque_Nm(remove) = [];
	
	% Remove low speed low load points
	remove = speed_radps < speed_scale*0.25 & torque_Nm < 0.4* torque_scale ;
	speed_radps(remove) = [];
	torque_Nm(remove) = [];
	
	% REmove points below provided WOT
	if ~isempty( WOT) && numel(WOT.speed_radps) > 1
		remove =  torque_Nm < interp1( WOT.speed_radps, WOT.torque_Nm, speed_radps,'linear');
		speed_radps(remove) = [];
		torque_Nm(remove) = [];
	end
	

	% Curve speeds should be monotonic and increasing
	for idx = (length(speed_radps) -1):-1:1
		speed_radps(idx) = min(speed_radps(idx), speed_radps(idx+1) - max( eps, 10* (torque_Nm(idx+1) - torque_Nm(idx))/ torque_scale )); 
	end
	
end

% Begin constructing curve by selecting points with largest error
[speed_radps,torque_Nm] = build_curve(speed_radps,torque_Nm, tol);

% Cleanup zero speed
if torque_Nm(1) ~= 0 || speed_radps(1) ~= 0
	torque_Nm = [0;torque_Nm(speed_radps>0)];
	speed_radps = [0;speed_radps(speed_radps>0)];
end

% Roll off on top end
if torque_Nm(end) > 0
	speed_radps = [speed_radps(:); speed_radps(end) .* [1.05; 1.10] ];
	torque_Nm = interp1( speed_radps(1:end-2), torque_Nm(:), speed_radps, 'linear','extrap' );
	torque_Nm(end-1) = min( torque_Nm(end-1), 0.90*torque_Nm(end-2 ) * speed_radps(end-2) / speed_radps(end-1) );		% Force downward slope / No Increase in power
	torque_Nm(end) = min(torque_Nm(end), 0.0 );								% Force end to Zero If not already
end

end


function [speed_radps, torque_Nm] = process_closed_throttle( CT, core, max_speed_radps, displacement_L,  tol, user_alpha)


if ~isempty(CT) && length( CT.speed_radps ) > 10 
		
	% Filter provided data by averaging into bins
	[speed_radps,torque_Nm] =  gen_avg_curve(CT.speed_radps, CT.torque_Nm, 50 * unit_convert.rpm2radps);
			
elseif ~isempty(core) && (sum( core.torque_Nm < 0 ) > 5 || ~isempty(CT))
	
	if ~isempty( CT)
		speed_pts_radps = [core.speed_radps;CT.speed_radps;0];
		torque_pts_Nm = [core.torque_Nm;CT.torque_Nm;0];
	else
		speed_pts_radps = [core.speed_radps;0];
		torque_pts_Nm = [core.torque_Nm;0];
	end
	
	% Negative Torques were mapped - Use Convex Hull
% 	xscat = [core.speed_radps;0];
% 	yscat = [core.torque_Nm;0];
	
	speed_scale = max(speed_pts_radps);
	torque_scale = max(torque_pts_Nm);
	
	% Scale and keep Unique
	speed_pts_norm = speed_pts_radps / speed_scale;
	torque_pts_norm = torque_pts_Nm / torque_scale;
	
	% Keep just uniques
	[unique_pts_norm, unique_in_idx] = unique( [speed_pts_norm , torque_pts_norm],'rows');

	
	%Compute ALpha Shape
	shp = alphaShape(unique_pts_norm);
	crit_alpha = shp.criticalAlpha('one-region');
	shp.Alpha = max( crit_alpha, user_alpha);

	bound_idx  = shp.boundaryFacets();
	bound_idx = bound_idx([1,end:-1:1],1);
% 	bound_idx(end+1) = bound_idx(1);

	speed_radps = speed_scale * speed_pts_norm(unique_in_idx(bound_idx));
	torque_Nm = torque_scale * torque_pts_norm(unique_in_idx(bound_idx));

	vert = [false; ( abs(diff( speed_radps)) <= 2.5e-3*speed_scale & diff( torque_Nm) > 0) ] | [  (abs(diff( speed_radps)) <= 1e-3*speed_scale & diff( torque_Nm) < 0); false ];
	remove = speed_radps > speed_scale | torque_Nm >= -0.05 * torque_scale |  vert;
	
	speed_radps(remove) = [];
	torque_Nm(remove) = [];
	
	for idx = 2:length(speed_radps)
		speed_radps(idx) = min(speed_radps(idx), speed_radps(idx-1)); 
    end

else
    % displacement based
    % Scale "default" closed throttle curve by engine engine.displacement_L
	speed_radps = unit_convert.rpm2radps*[0 657.2 812.1 1050 1991 3008 4064 4555 7000];
	torque_Nm = [-26.32, -30.57 -32.01 -34.12 -39.84 -45.98 -52.72 -55.77,-71.58] * displacement_L / 3.3;

end

% Construct curve by selecting points with largest error
[speed_radps,torque_Nm] = build_curve(speed_radps,torque_Nm, tol);
	
% Extend from 0 to max WOT speed

% Use end points to extrapolate;
extrap_speed_radps = 	speed_radps([1,end]);
extrap_torque_Nm =		torque_Nm([1,end]);
	
if speed_radps(1) > 0
	min_torque_pt_Nm = max( torque_Nm(1), interp1(extrap_speed_radps, extrap_torque_Nm, 0,'linear','extrap'));
	speed_radps = [0; speed_radps(:)];
	torque_Nm = [min_torque_pt_Nm; torque_Nm(:)];
end

if speed_radps(end) < max_speed_radps
	min_torque_pt_Nm = min( torque_Nm(end), interp1(extrap_speed_radps, extrap_torque_Nm, max_speed_radps,'linear','extrap'));
	speed_radps = [speed_radps(:); max_speed_radps];
	torque_Nm = [ torque_Nm(:); min_torque_pt_Nm];
end
	
end


function [speed_radps, torque_Nm] = adjust_closed_throttle(speed_radps, torque_Nm,  map_speed_radps, map_torque_Nm, map_fuel_gps, max_speed_radps, tol, varargin)

combined_speeds_radps = unique([ speed_radps(:); map_speed_radps(:)]);
fuel_gps = interp1( map_speed_radps, map_fuel_gps', combined_speeds_radps,'linear','extrap');	
	
% Limit to point where fuel from map is zero fuel

adj_torque_Nm = ones(size(combined_speeds_radps)) .* torque_Nm(1);

if strcmpi( 'disable_zero_fuel_corr_CT', varargin)
    for s = 2:length(combined_speeds_radps)
        zero_fuel_torque = interp1( fuel_gps(s,:), map_torque_Nm, 0.0,'spline','extrap');
        orig_torque = interp1(speed_radps, torque_Nm,combined_speeds_radps(s),'linear','extrap');
        adj_torque_Nm(s) = min( adj_torque_Nm(s-1), orig_torque);
    end
    
else
    for s = 2:length(combined_speeds_radps)
        zero_fuel_torque = interp1( fuel_gps(s,:), map_torque_Nm, 0.0,'spline','extrap');
        orig_torque = interp1(speed_radps, torque_Nm,combined_speeds_radps(s),'linear','extrap');
        adj_torque_Nm(s) = min( adj_torque_Nm(s-1), max( zero_fuel_torque, orig_torque));
    end
end

% Construct curve by selecting points with largest error
[speed_radps,torque_Nm] = build_curve(combined_speeds_radps,adj_torque_Nm, tol);
	
% Extend from 0 to max WOT speed

% Use end points to extrapolate;
extrap_speed_radps = 	speed_radps([1,end]);
extrap_torque_Nm =		torque_Nm([1,end]);
	
if speed_radps(1) > 0
	min_torque_pt_Nm = max( torque_Nm(1), interp1(extrap_speed_radps, extrap_torque_Nm, 0,'linear','extrap'));
	speed_radps = [0; speed_radps(:)];
	torque_Nm = [min_torque_pt_Nm; torque_Nm(:)];
end

if speed_radps(end) < max_speed_radps
	min_torque_pt_Nm = min( torque_Nm(end), interp1(extrap_speed_radps, extrap_torque_Nm, max_speed_radps,'linear','extrap'));
	speed_radps = [speed_radps(:); max_speed_radps];
	torque_Nm = [ torque_Nm(:); min_torque_pt_Nm];
end
	
end


function out = convert_and_combine(in, displacement_L, has_fuel)

if isempty(in)
	out = [];
	return;
end

if ~iscell(in)
	in = {in};
end


out.speed_radps = [];
out.torque_Nm = [];
out.dataset = [];

out.fuel_gps = [];
out.bsfc_gpkWhr = [];

out.manifold_press_kPaA = [];

for i = 1:length(in)
	
	% Generate speed in radps if missing
	if ~isfield( in{i},'speed_radps') && isfield(in{i}, 'speed_rpm' )
		in{i}.speed_radps = in{i}.speed_rpm * unit_convert.rpm2radps;
	end
	
	% Generate torque in Nm if missing
	if ~isfield( in{i},'torque_Nm') && isfield( in{i},'bmep_bar')
		in{i}.torque_Nm = in{i}.bmep_bar / ( 4 * pi / (displacement_L /1000)*1e-5 );
    end
	
    % Generate torque in Nm if missing by using imep and CT torque 
	if ~isfield( in{i},'torque_Nm') && isfield( in{i},'imep_bar')
        
        % Estimate closed throttle curve if not available       	
        if ~isfield( in{i},'CT')
            [CT_speed_radps, CT_torque_Nm] = process_closed_throttle( [], [], max(in{i}.speed_rpm * unit_convert.rpm2radps), displacement_L,  20, []); % any value more than 0 would work, as we don't know max WOT torque
        end
        
        % Estimate Friction torque
            in{i}.ind_torque_Nm = (in{i}.imep_bar / ( 4 * pi / (displacement_L /1000)*1e-5 ));
            in{i}.fri_torque_Nm = interp1(CT_speed_radps,CT_torque_Nm,in{i}.speed_radps,'linear');
            in{i}.torque_Nm =  in{i}.ind_torque_Nm + in{i}.fri_torque_Nm; % '+' here because friction is a negative value
            
            % exclude those data points where the in{i}.ind_torque_Nm < in{i}.fri_torque_Nm; the cause
            % may be due to the error in estimating friction torque based on the look-up table 
                keep_Idx = find((in{i}.ind_torque_Nm + in{i}.fri_torque_Nm) > 0);

                in{i}.speed_radps = in{i}.speed_radps(keep_Idx);
                in{i}.speed_rpm = in{i}.speed_rpm(keep_Idx);
                in{i}.torque_Nm = in{i}.torque_Nm(keep_Idx);
                in{i}.imep_bar = in{i}.imep_bar(keep_Idx);
                in{i}.bsfc_gpkWhr = in{i}.bsfc_gpkWhr(keep_Idx);
                in{i}.ind_torque_Nm = in{i}.ind_torque_Nm(keep_Idx);
                in{i}.fri_torque_Nm = in{i}.fri_torque_Nm(keep_Idx);
    end
    
	
	
	% Serialize fuel if table provided
	if isfield( in{i},'fuel_gps') && ~isvector( in{i}.fuel_gps)
		[in{i}.fuel_gps, in{i}.torque_Nm, in{i}.speed_radps] = table2columns(  in{i}.fuel_gps, in{i}.torque_Nm, in{i}.speed_radps );
	end
	
	% Serialize BSFC if table provided
	if isfield( in{i},'bsfc_gpkWhr') && ~isvector( in{i}.bsfc_gpkWhr)
		[in{i}.bsfc_gpkWhr, in{i}.torque_Nm, in{i}.speed_radps] = table2columns(  in{i}.bsfc_gpkWhr, in{i}.torque_Nm, in{i}.speed_radps );
	end
	
	% Serialize BTE if table provided
	if isfield( in{i},'efficiency_pct') && ~isvector( in{i}.efficiency_pct)
		[in{i}.efficiency_pct, in{i}.torque_Nm, in{i}.speed_radps] = table2columns(  in{i}.efficiency_pct, in{i}.torque_Nm, in{i}.speed_radps );
	end
	
	
	
	% Generate fuel in g/sec and BSFC if missing
	if isfield( in{i},'fuel_gps')
		% Have fuel data directly - Need to calc BSFC
		in{i}.bsfc_gpkWhr = in{i}.fuel_gps ./ ( in{i}.speed_radps  .* in{i}.torque_Nm /1000 / 3600 );
	elseif isfield( in{i},'bsfc_gpkWhr')
		% Calculate from BSFC
		in{i}.fuel_gps = in{i}.bsfc_gpkWhr .* in{i}.speed_radps  .* in{i}.torque_Nm /1000 / 3600;
	elseif isfield( in{i},'efficiency_pct')
		% Calculate from Efficiency
		in{i}.fuel_gps = (in{i}.speed_radps  .* in{i}.torque_Nm) ./( in{i}.efficiency_pct / 100 .* in{i}.fuel_prop.energy_density_MJpkg *1000 );
		in{i}.bsfc_gpkWhr = 3600./( in{i}.efficiency_pct / 100 .* in{i}.fuel_prop.energy_density_MJpkg  ) ;
	else
		% Don't know - fill with NaN
		in{i}.fuel_gps = nan(size(in{i}.speed_radps));
		in{i}.bsfc_gpkWhr = nan(size(in{i}.speed_radps));
	end
	
	
	% Generate MAP in kPa
	if isfield( in{i},'manifold_press_kPaA')
		% Already Have it
	else
		in{i}.manifold_press_kPaA = nan(size(in{i}.speed_radps));
	end
	
	
	

	% Store dataset Name
	if isfield( in{i},'name' )
		out.names{i} = in{i}.name;
	else
		out.names{i} = sprintf('Dataset %d',i);
	end

	
	%Concatenate data
	out.dataset = [out.dataset; i * ones(size(in{i}.speed_radps(:)))];
	
	out.speed_radps = [out.speed_radps; in{i}.speed_radps(:)];
	out.torque_Nm = [out.torque_Nm; in{i}.torque_Nm(:)];
	
	out.fuel_gps = [out.fuel_gps; in{i}.fuel_gps(:)];
	out.bsfc_gpkWhr = [out.bsfc_gpkWhr; in{i}.bsfc_gpkWhr(:)];
	
	out.manifold_press_kPaA = [out.manifold_press_kPaA; in{i}.manifold_press_kPaA(:)];
	
end


% Remove Nan speed or torque - other fields checked separately
nan_idx = isnan(out.torque_Nm) | isnan(out.speed_radps) ;

%     if isfield( in{i},'fuel_gps')
%         nan_idx = nan_idx | isnan(in{i}.fuel_gps);
%         in{i}.fuel_gps(nan_idx) = [];
% 		in{i}.bsfc_gpkWhr(nan_idx) = [];
%     end

out.speed_radps(nan_idx) = [];
out.torque_Nm(nan_idx) = [];
out.dataset(nan_idx) = [];

out.fuel_gps(nan_idx) = [];
out.bsfc_gpkWhr(nan_idx) = [];

out.manifold_press_kPaA(nan_idx) = [];


end


function [spd, trq] = gen_avg_curve( spd_in, trq_in, step)

spd = 0:step:max(spd_in);
trq = zeros(size(spd));

i = length( spd);
while i > 0
	pts = abs(spd_in - spd(i)) <= step/2 ;
	
	if sum(pts) <= 0
		trq(i) = [];
		spd(i) = [];
		
	else
		trq(i) = mean( trq_in(pts ) );
		spd(i) = mean( spd_in(pts ) );
		
	end
	
	i = i-1;
	
end
end

function out = fill_gaps( data, max_gap )

data = unique(data(:));

big_gap_left = diff(data) > max_gap;
big_gap_left(end+1) = false; % Tack on end to match size
big_gap_right = circshift(big_gap_left, 1);

add_points = (data(big_gap_left) + data(big_gap_right))/2;

out = unique([data; add_points(:)]);

end


function [spd, trq] = build_curve(spd_in, trq_in, tol )

spd_in = spd_in(:);
trq_in = trq_in(:);

[spd_in, sort_idx] = unique(spd_in);
trq_in = trq_in(sort_idx);

deriv = (diff( trq_in) ./ max(eps, diff( spd_in))) / (100 * tol);
keep = [true ; abs(diff(deriv)) > 0.002 ; true];

spd = spd_in(keep);
trq = trq_in(keep);

max_err = inf;
while max_err > tol
	
	err = trq_in - interp1(spd, trq, spd_in,'linear');
	
	[max_err, idx] = max( abs(err));
	
	spd(end+1) = spd_in(idx);
	trq(end+1) = trq_in(idx);
	
	[spd, sort_idx] = unique(spd);
	trq = trq(sort_idx);
	
end

end

