function [delta, speeds_radps, torques_Nm] = REVS_calc_engine_diff(e1, e2, compare_mode, varargin)
%function REVS_plot_engine_efficiency(e, varargin)
%
% no_figure                   = parse_varargs(varargin,'no_figure',false,'toggle');
% contour_levels              = parse_varargs(varargin,'contours',[0,10, 15, 20, 23, 25:50],'numeric',{'vector'});
% contour_filled              = parse_varargs(varargin,'filled',false,'toggle');
% contours_labels             = parse_varargs(varargin,'no_contour_labels',true,'toggle');
% contour_width               = parse_varargs(varargin,'line_width',1,'numeric');
% contour_color               = parse_varargs(varargin,'line_color','auto');
% power_lines                 = parse_varargs(varargin,'no_power_lines',true,'toggle');
% cmap                        = parse_varargs(varargin,'colormap','jet');
% highlight_patch             = parse_varargs(varargin,'highlight_contour',[],'numeric');
% highlight_color             = parse_varargs(varargin,'highlight_color',[0,.6,0],'numeric');
% highlight_alpha             = parse_varargs(varargin,'highlight_alpha',[],'numeric',{'scalar'});
% full_throttle_width         = parse_varargs(varargin,'line_width',3,'numeric');
% fill_above_full_throttle    = parse_varargs(varargin,'showabovefullthrottle',true,'toggle');
% plot_audit                  = parse_varargs(varargin,'plot_audit',false,'');


if nargin < 3
	compare_mode				= 'efficiency';
else
	validatestring(compare_mode,{'efficiency','bsfc','bsfc_pct','gps','gps_pct','fuel','fuel_pct', 'fuel_vol','fuel_vol_pct','co2','co2_pct' });
end

correct_fuel				= parse_varargs(varargin,'no_fuel_adjust',true,'toggle');


scaling_mode				= parse_varargs(varargin,'scaling','avg_power','char',{'none','max_power','max_torque','avg_torque','avg_power'});
scaling_window				= parse_varargs( varargin,'scaling_window',linspace(1800, 5500, 25)* unit_convert.rpm2radps, 'numeric' );
scaling_fuel				= parse_varargs( varargin,'scaling_no_fuel_adjust', true, 'toggle');

speeds_radps					= parse_varargs( varargin,'speed_axis', e1.fuel_map_speed_radps, 'numeric');
torques_Nm					= parse_varargs( varargin,'torque_axis', e1.fuel_map_torque_Nm, 'numeric');




% Get statistics on engine 1
e1_torque = max(0,interp1( e1.full_throttle_speed_radps, e1.full_throttle_torque_Nm, scaling_window,'linear','extrap' ));
e1_average_torque = sum(e1_torque )./ length(scaling_window);
e1_average_power = sum(e1_torque .* scaling_window )./ length(scaling_window);


if correct_fuel
	e2.fuel_map_gps = e2.fuel_map_gps * e2.fuel.energy_density_MJpkg / e1.fuel.energy_density_MJpkg;
end

opts = {};

if ~scaling_fuel
% 	opts{end+1} = 'no_scale_fuel_adjust'; % For old REVS_modify_engine
	opts{end+1} = 'no_scaling_knock_adjust';
	opts{end+1} = 'no_scaling_friction_adjust';
	opts{end+1} = 'no_scaling_sv_ratio_adjust';
end


% Scale 2nd engine if needed
switch lower(scaling_mode)
	
	case 'max_power'
		e2 = REVS_modify_engine2( e2, 'scale_max_power', e1.max_power_kW, opts{:});
	case 'max_torque'
		e2 = REVS_modify_engine2( e2, 'scale_max_torque', e1.max_torque_Nm, opts{:});
	case 'avg_power'
		e2 = REVS_modify_engine2( e2, 'scale_avg_power', e1_average_power, 'avg_window', scaling_window, opts{:});
	case 'avg_torque'
		e2 = REVS_modify_engine2( e2, 'scale_avg_torque', e1_average_torque, 'avg_window', scaling_window, opts{:});	
		
end


[mesh_speed_radps, mesh_torque_Nm] = meshgrid( speeds_radps, torques_Nm);

%Compute Gridded Maps for comparison
[e1_gps, e1_eff, e1_bsfc] = interp_engine( e1, mesh_speed_radps, mesh_torque_Nm );
[e2_gps, e2_eff, e2_bsfc] = interp_engine( e2, mesh_speed_radps, mesh_torque_Nm );


switch lower(compare_mode)
	case 'efficiency'
		delta =  e1_eff - e2_eff;
	case 'bsfc'
		delta =  e1_bsfc - e2_bsfc;
	case {'gps','fuel'}
		delta =  e1_gps - e2_gps;
	case 'bsfc_pct'
		delta = 100 * (e1_gps - e2_gps) ./ e2_gps;
	case {'gps_pct','fuel_pct'}
		delta = 100 * (e1_gps - e2_gps) ./ e2_gps;
    case 'fuel_vol'
		delta = e1_gps./e1.fuel.density_kgpL_15C - e2_gps./e2.fuel.density_kgpL_15C;
	case 'fuel_vol_pct'
		delta = 100*(e1_gps./e1.fuel.density_kgpL_15C - e2_gps./e2.fuel.density_kgpL_15C) ./ (e2_gps./e2.fuel.density_kgpL_15C);
    case 'co2'
		delta = (e1_gps.*e1.fuel.carbon_weight_fraction - e2_gps.*e2.fuel.carbon_weight_fraction).*(44.0095/12.0107);
	case 'co2_pct'
		delta = 100*(e1_gps.*e1.fuel.carbon_weight_fraction - e2_gps.*e2.fuel.carbon_weight_fraction)./(e2_gps.*e2.fuel.carbon_weight_fraction);
    otherwise
		error( 'Unknown comparison mode');
		
end



end


function [gps, eff, bsfc] = interp_engine( e, mesh_speed_radps, mesh_torque_Nm )

gps = interp2( e.fuel_map_speed_radps,  e.fuel_map_torque_Nm, e.fuel_map_gps , mesh_speed_radps, mesh_torque_Nm);

% Nan out points above WOT
% ft = interp1(e.full_throttle_speed_radps, e.full_throttle_torque_Nm, mesh_speed_radps) + 0.05*max(e.full_throttle_torque_Nm);
% gps( mesh_torque_Nm > ft) = nan;



bsfc = gps * 3600 ./ max(eps, (mesh_speed_radps .* mesh_torque_Nm) ./ 1000 );


eff = 100 * ((mesh_speed_radps .* mesh_torque_Nm) ./ 1000 ) ./ ( gps * e.fuel.energy_density_MJpkg );


end




