function [ emachine_out ] = REVS_modify_emachine( emachine_in, varargin )
%REVS_MODIFY_EMACHINE takes in an eMACHINE structure, applies emachine modifiers,
%   and returns the result in a new emachine structure.
%
%   [ emachine_new ] = REVS_MODIFY_EMACHINE( engine, varargin )
%
%   Examples
%
%       Scale emachine to 50kW emachine:
%       [ scaled_emachine ] = REVS_MODIFY_EMACHINE( emotor, 'scale_max_torque', 50 )
%
%       Scale engine to 50 kW, 100 Nm, 12000 rpm:
%       [ scaled_engine ] = REVS_MODIFY_EMACHINE( engine, 'scale_max_torque', 50, 'scale_max_torque', 100 ,'scale_max_speed',12000 )
%
%   Avaiable emachine modifiers can be applied by varargin as follows:
%     max_motor_power = parse_varargs( varargin, 'scale_max_torque', [], 'numeric' );
%
%   see also parse_varargs.

%% Parse varargs
% Check if enough varargin was passed
narginchk( 1, inf );

% Validate required arguments
validate_arg( emachine_in, {'class_REVS_emachine','struct'});

% emachine Scaling Options

% scale_factor = parse_varargs( varargin, 'scale_factor', [], 'numeric' );
% scale_displacement = parse_varargs( varargin, 'scale_displacement', [], 'numeric' );
scale_max_power = parse_varargs( varargin, 'scale_max_power_kW',[], 'numeric' );
% scale_avg_power = parse_varargs( varargin, 'scale_avg_power',[], 'numeric' );
scale_max_torque = parse_varargs( varargin, 'scale_max_torque_Nm',[], 'numeric' );
% scale_avg_torque = parse_varargs( varargin, 'scale_avg_torque',[], 'numeric' );
scale_max_speed = parse_varargs( varargin, 'scale_max_speed_rpm',[], 'numeric' );

gear_ratio = parse_varargs( varargin, 'gear_ratio',[], 'numeric' );

% Unpowered Torque Adjustment
% input_friction_adjust_Nm = parse_varargs( varargin, 'friction_adjust_Nm', 0.0, 'numeric');
% input_friction_adjust_bar = parse_varargs( varargin, 'friction_adjust_bar', 0.0, 'numeric');

%% Copy and create a new engine structure
emachine_out = emachine_in;
% emachine_out = class_REVS_emachine_geared;


%% Compute Desired Scaling
if ~isempty(scale_max_power)
        scale_factor.power = scale_max_power*1000/emachine_in.max_motor_power_W;
                
    if ~isempty(scale_max_torque)
        scale_factor.torque = scale_max_torque/emachine_in.max_torque_Nm;
        scale_factor.speed = (scale_max_power*1000/scale_max_torque)/(emachine_in.max_motor_power_W/emachine_in.max_torque_Nm);
        emachine_out.name = sprintf([emachine_in.source_filename , ' scaled to a %1.0f kW %1.0f Nm machine'], scale_max_power, scale_max_torque);
    
    else
        scale_factor.torque = (scale_factor.power);
        scale_factor.speed = 1;
        emachine_out.name = sprintf([emachine_in.source_filename , ' scaled to a %1.0f kW'], scale_max_power);
  
           
    end

    
    %Update electrical consumption values   
    
    emachine_out.electric_power_W.axis_1.breakpoints =  emachine_in.electric_power_W.get_signal_breakpoints('emach_trq_Nm').*scale_factor.torque;    
    emachine_out.electric_power_W.axis_2.breakpoints =  emachine_in.electric_power_W.get_signal_breakpoints('emach_spd_radps').*scale_factor.speed;
    

    speeds_radps = emachine_out.electric_power_W.get_signal_breakpoints( 'emach_spd_radps');
    torque_Nm = emachine_out.electric_power_W.get_signal_breakpoints( 'emach_trq_Nm');
    voltage_v = emachine_out.electric_power_W.get_signal_breakpoints( 'emach_voltage_V');
    
    if ~isempty(voltage_v)
        [mesh_speed_radps,mesh_torque_Nm,~] = meshgrid(speeds_radps,torque_Nm,voltage_v);
    else
        [mesh_speed_radps,mesh_torque_Nm] = meshgrid(speeds_radps,torque_Nm);
    end
    
  	mesh_mech_power = mesh_speed_radps.*mesh_torque_Nm;
    
%     emachine_out.electric_power_W.table = mesh_mech_power ./eff_map_norm;
%     emachine_out.electric_power_W.table(mesh_mech_power < 0 ) = mesh_mech_power(mesh_mech_power <0 ).*eff_map_norm(mesh_mech_power <0);   
%     idx = abs(mesh_mech_power) == 0 & mesh_speed_radps == 0 ;

%   Scale electrical power consumption based on power scaling factor
    emachine_out.electric_power_W.table = emachine_in.electric_power_W.table.*scale_factor.power ;
    
    
    %Modify limits

    
    emachine_out.positive_torque_limit_Nm.table(:) = emachine_in.positive_torque_limit_Nm.table(:).*scale_factor.torque;
    emachine_out.negative_torque_limit_Nm.table(:) = emachine_in.negative_torque_limit_Nm.table(:).*scale_factor.torque;
    
    emachine_out.positive_torque_limit_Nm.axis_1.breakpoints(:) = emachine_in.positive_torque_limit_Nm.axis_1.breakpoints(:).*scale_factor.speed;
    emachine_out.negative_torque_limit_Nm.axis_1.breakpoints(:) = emachine_in.negative_torque_limit_Nm.axis_1.breakpoints(:).*scale_factor.speed;
    
    emachine_out.max_torque_Nm = emachine_in.max_torque_Nm.*scale_factor.torque;
    
    emachine_out.max_motor_power_W = scale_max_power*1000;
    emachine_out.max_generator_power_W = scale_max_power*1000;
    
%     emachine_out.name = sprintf([emachine_in.source_filename , ' scaled to a %1.0f kW - %1.0f Nm machine'], scale_max_power);
end

%% Adjust Physical Characteristics
if ~isempty(gear_ratio) && contains(class(emachine_in),'class_REVS_emachine_geared')
    emachine_out.gear.ratio = gear_ratio;
    
elseif ~isempty(gear_ratio) && contains(class(emachine_in),'class_REVS_emachine')
    emachine_out2 = class_REVS_emachine_geared;
    emachine_out2 = copyObject(emachine_out,emachine_out2);
    emachine_out2.gear.ratio = gear_ratio;
    clear emachine_out
    emachine_out = copyObject(emachine_out2,emachine_out);
end



end


function output = copyObject(input, output)
 C = metaclass(input);
 P = C.Properties;
 for k = 1:length(P)
   if ~P{k}.Dependent
     output.(P{k}.Name) = input.(P{k}.Name);
   end
 end
end
