function [ engine_out , engine_modifiers ] = REVS_modify_engine( engine_in, varargin )
%REVS_MODIFY_ENGINE2 takes in an engine structure and its displacement(L), apply engine modifiers,
%   and returns the result in a new engine structure.
%
%   [ engine_new ] = REVS_MODIFY_ENGINE( engine, varargin )
%
%   Examples
%   
%       Scale engine to 3.6L engine:
%       [ scaled_engine ] = REVS_MODIFY_ENGINE( engine, 'scale_displacement', 3.6 )
%
%       Scale engine to 3.6L engine without knock adjustment:
%       [ scaled_engine ] = REVS_MODIFY_ENGINE( engine, 'scale_displacement', 3.6, 'no_scaling_knock_adjust' )
%
%   Avaiable engine modifiers can be applied by varargin as follows:
%     num_cylinders = parse_varargs( varargin, 'num_cylinders', [], 'numeric' );
% 
%     scale_factor = parse_varargs( varargin, 'scale_factor', [], 'numeric' );
%     scale_displacement = parse_varargs( varargin, 'scale_displacement', [], 'numeric' );
%     scale_max_power = parse_varargs( varargin, 'scale_max_power',[], 'numeric' );
%     scale_avg_power = parse_varargs( varargin, 'scale_avg_power',[], 'numeric' );
%     scale_max_torque = parse_varargs( varargin, 'scale_max_torque',[], 'numeric' );
%     scale_avg_torque = parse_varargs( varargin, 'scale_avg_torque',[], 'numeric' );
% 
%     limit_bmep = parse_varargs( varargin, 'limit_bmep',[], 'numeric' );
%     limit_torque = parse_varargs( varargin, 'limit_torque',[], 'numeric' );
% 
%     % window to use for averaging power or torque
%     averaging_window = parse_varargs( varargin, 'avg_window',[1800,6000] * unit_convert.rpm2radps, 'numeric' );
% 
%     scaling_knock_adjust    = parse_varargs( varargin,'no_scaling_knock_adjust',   true,'toggle');
%     scaling_friction_adjust = parse_varargs( varargin,'no_scaling_friction_adjust',true,'toggle');
%     scaling_sv_ratio_adjust = parse_varargs( varargin,'no_scaling_sv_ratio_adjust',true,'toggle');
% 
%     % Fricion Adjustment
%     input_friction_adjust_Nm = parse_varargs( varargin, 'friction_adjust_Nm', 0.0, 'numeric');
%     input_friction_adjust_bar = parse_varargs( varargin, 'friction_adjust_bar', 0.0, 'numeric');
% 
%     knk_max_adjust = parse_varargs( varargin, 'knock_max_adjust', 1.1, 'numeric');
% 
%     % Add or remove engine technologies
%     rem_deac = parse_varargs( varargin,'-deac', false, 'toggle');
%     add_deacC = parse_varargs( varargin,'+deacC', false, 'toggle');
%     add_deacD = parse_varargs( varargin,'+deacD', false, 'toggle');
%     deac_num_cylinders = parse_varargs( varargin,'deac_cylinders', [], 'numeric','scalar');
%     deac_activation_time = parse_varargs( varargin,'deac_activation_secs', 120, 'numeric','scalar');
%     deac_reduction_curve = parse_varargs( varargin,'deac_reduction_curve', [], 'struct');
%     deac_max_reduction = parse_varargs( varargin,'deac_max_reduction', [], 'numeric');
%     deac_min_speed_radps = parse_varargs( varargin,'deac_min_speed_radps', 940 * unit_convert.rpm2radps, 'numeric','scalar');
%     deac_max_speed_radps = parse_varargs( varargin,'deac_max_speed_radps', 4000 * unit_convert.rpm2radps, 'numeric','scalar');
% 
%     mod_GDI = parse_varargs( varargin,'GDI', 0, 'numeric');           % 0..1 to add, 0..-1 to remove, in proportion
%     mod_DCP_DVVL = parse_varargs( varargin,'DCP_DVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
%     mod_CCP_DVVL = parse_varargs( varargin,'CCP_DVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
%     mod_DCP_CVVL = parse_varargs( varargin,'DCP_CVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
%     mod_DCP      = parse_varargs( varargin,'DCP', 0, 'numeric');      % 0..1 to add, 0..-1 to remove, in proportion
%     mod_CCP      = parse_varargs( varargin,'CCP', 0, 'numeric');      % 0..1 to add, 0..-1 to remove, in proportion
% 
%     new_fuel     = parse_varargs( varargin,'fuel', []); % pass in class_REVS_fuel object
%
%   see also parse_varargs.

%% Parse varargs
% Check if enough varargin was passed
narginchk( 1, inf );

% Validate required arguments
validate_arg( engine_in, {'class_REVS_engine','class_REVS_engine_legacy','struct'});

% Engine Scaling Options
num_cylinders = parse_varargs( varargin, 'num_cylinders', [], 'numeric' );

scale_factor = parse_varargs( varargin, 'scale_factor', [], 'numeric' );
scale_displacement = parse_varargs( varargin, 'scale_displacement', [], 'numeric' );
scale_max_power = parse_varargs( varargin, 'scale_max_power',[], 'numeric' );
scale_avg_power = parse_varargs( varargin, 'scale_avg_power',[], 'numeric' );
scale_max_torque = parse_varargs( varargin, 'scale_max_torque',[], 'numeric' );
scale_avg_torque = parse_varargs( varargin, 'scale_avg_torque',[], 'numeric' );

limit_bmep = parse_varargs( varargin, 'limit_bmep',[], 'numeric' );
limit_torque = parse_varargs( varargin, 'limit_torque',[], 'numeric' );

% window to use for averaging power or torque
averaging_window = parse_varargs( varargin, 'avg_window',[1800,6000] * unit_convert.rpm2radps, 'numeric' );

scaling_knock_adjust    = parse_varargs( varargin,'no_scaling_knock_adjust',   true,'toggle');
scaling_friction_adjust = parse_varargs( varargin,'no_scaling_friction_adjust',true,'toggle');
scaling_sv_ratio_adjust = parse_varargs( varargin,'no_scaling_sv_ratio_adjust',true,'toggle');


% Fricion Adjustment
input_friction_adjust_Nm = parse_varargs( varargin, 'friction_adjust_Nm', 0.0, 'numeric');
input_friction_adjust_bar = parse_varargs( varargin, 'friction_adjust_bar', 0.0, 'numeric');

knk_max_adjust = parse_varargs( varargin, 'knock_max_adjust', 1.1, 'numeric');

% Add or remove engine technologies
rem_deac = parse_varargs( varargin,'-deac', false, 'toggle');
add_deacC = parse_varargs( varargin,'+deacC', false, 'toggle');
add_deacD = parse_varargs( varargin,'+deacD', false, 'toggle');
deac_method = parse_varargs( varargin, 'deac_method','reduction curve','char',{'reduction curve','ind_eff'});
deac_num_cylinders = parse_varargs( varargin,'deac_cylinders', [], 'numeric','scalar');
deac_activation_time = parse_varargs( varargin,'deac_activation_secs', 120, 'numeric','scalar');
deac_reduction_curve = parse_varargs( varargin,'deac_reduction_curve', [], 'struct');
deac_max_reduction = parse_varargs( varargin,'deac_max_reduction', [], 'numeric');
deac_min_speed_radps = parse_varargs( varargin,'deac_min_speed_radps', 940 * unit_convert.rpm2radps, 'numeric','scalar');
deac_max_speed_radps = parse_varargs( varargin,'deac_max_speed_radps', 4000 * unit_convert.rpm2radps, 'numeric','scalar');

mod_GDI = parse_varargs( varargin,'GDI', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
mod_DCP_DVVL = parse_varargs( varargin,'DCP_DVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
mod_CCP_DVVL = parse_varargs( varargin,'CCP_DVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
mod_DCP_CVVL = parse_varargs( varargin,'DCP_CVVL', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
mod_DCP      = parse_varargs( varargin,'DCP', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion
mod_CCP      = parse_varargs( varargin,'CCP', 0, 'numeric'); % 0..1 to add, 0..-1 to remove, in proportion

new_fuel     = parse_varargs( varargin,'fuel', []); % pass in class_REVS_fuel object
fuel_octane_adjust = parse_varargs( varargin,'no_fuel_octane_adjust',true,'toggle');

% Fill in if only window end points provided
if numel(averaging_window) ==2
    averaging_window = linspace(averaging_window(1), averaging_window(end), 25);
end



%% Copy and create a new engine structure
engine_out = engine_in;

%% Compute Desired Scaling

engine_torque = interp1( engine_in.full_throttle_speed_radps, engine_in.full_throttle_torque_Nm, averaging_window,'linear','extrap' );
engine_power = engine_torque .* averaging_window;

engine_average_torque = sum( engine_torque)./ length(averaging_window);
engine_average_power = sum( engine_power )./ length(averaging_window);


if ~isempty( scale_displacement )
    scale_factor = scale_displacement / engine_in.displacement_L;
elseif ~isempty( scale_max_power )
    scale_factor = scale_max_power / engine_in.max_power_W;
elseif ~isempty( scale_max_torque )
    scale_factor = scale_max_torque / engine_in.max_torque_Nm;
elseif ~isempty( scale_avg_power )
    scale_factor = scale_avg_power / engine_average_power;
elseif ~isempty( scale_avg_torque )
    scale_factor = scale_avg_torque / engine_average_torque;
elseif isempty(scale_factor) && ~isempty( num_cylinders )
    scale_factor = num_cylinders ./ engine_in.num_cylinders;
elseif isempty(scale_factor)
    scale_factor = 1.0;
end


%% Adjust Physical Characteristics
engine_out.displacement_L = engine_in.displacement_L * (scale_factor);
engine_out.inertia_kgm2	  = engine_in.inertia_kgm2 .* (scale_factor);

%% Set number of cylinders

disp_limit_cyl =     [	3,	  4,	  6,   	  8,	16];
disp_limit_max = 1.0*[1.5,	2.7,	4.5,	7.5,	15];
%disp_limit_max = [1.5,	2.7,	3.55,	7.5,	15.0];
disp_limit_min = 1.0*[0.6,	1.4,	2.5,	3.8,	8.0];
%disp_limit_min = [0.6,	1.0,	2.5,	3.56,	8.0];

if ~isempty(num_cylinders)
    % Use provided Cylinder Count
    engine_out.num_cylinders = num_cylinders;
elseif (scale_factor ~= 1.0)
    engine_out.num_cylinders = engine_in.num_cylinders;
    
    % Add cylinders is displacement above limit
    while engine_out.displacement_L	> interp1( disp_limit_cyl, disp_limit_max, engine_out.num_cylinders)
        engine_out.num_cylinders = engine_out.num_cylinders + 1 + (engine_out.num_cylinders >= 4);
    end
    
    % Remove cylinders if displacement below limit
    while engine_out.displacement_L	< interp1( disp_limit_cyl, disp_limit_min, engine_out.num_cylinders)
        engine_out.num_cylinders = engine_out.num_cylinders - 1 - (engine_out.num_cylinders >= 6);
    end
end

% Scale # of deac cylinders - may be non integer, will clean up later
engine_out.deac_num_cylinders = engine_in.deac_num_cylinders * engine_out.num_cylinders / engine_in.num_cylinders;

%% Generate cylinder scale factors
cyl_volume_scale_factor = scale_factor * engine_in.num_cylinders / engine_out.num_cylinders;
cyl_linear_scale_factor = cyl_volume_scale_factor ^ (1/3);

%% Adjust Cylinder Physical Characteristics
engine_out.bore_mm	  = engine_in.bore_mm .* cyl_linear_scale_factor;
engine_out.stroke_mm  = engine_in.stroke_mm .* cyl_linear_scale_factor;


if ~isempty( num_cylinders) || engine_out.num_cylinders ~= engine_in.num_cylinders
    engine_out.name = sprintf( '%s converted to %0.2fL %d Cylinder', engine_in.name, engine_out.displacement_L, engine_out.num_cylinders);
elseif scale_factor ~= 1
    engine_out.name = sprintf( '%s converted to %0.2fL', engine_in.name, engine_out.displacement_L);
end

% Make meshgrid of speed and torque for later use
[speed_mesh_radps, brake_torque_mesh_Nm] = meshgrid( engine_in.fuel_map_speed_radps, engine_in.fuel_map_torque_Nm);
ind_torque_mesh_Nm = brake_torque_mesh_Nm - interp1( engine_in.closed_throttle_speed_radps, engine_in.closed_throttle_torque_Nm, speed_mesh_radps,'linear','extrap');

above_wot = brake_torque_mesh_Nm > 1.05 * interp1( engine_in.full_throttle_speed_radps, engine_in.full_throttle_torque_Nm, speed_mesh_radps,'linear','extrap');

[min_consumption_speed_radps, min_consumption_torque_Nm] = REVS_calc_engine_min_bsfc(engine_in);
min_consumption_torque_Nm = cummax(min_consumption_torque_Nm);
min_consumption_power_W = min_consumption_speed_radps .* min_consumption_torque_Nm;
min_consumption_fuel_gps = interp2( engine_in.fuel_map_speed_radps, engine_in.fuel_map_torque_Nm, engine_in.fuel_map_gps, min_consumption_speed_radps, min_consumption_torque_Nm);


%% Friction Scaling

% SWRI: Friction Adjustment 0.1 bar FMEP for 10% Stroke increase - converted to torque
% Seemed big - try half
% 	scaling_friction_adjust_Nm =  scaling_friction_adjust * (cyl_stroke_scale_factor-1) * 100 * engine_out.displacement_L ./ (4 * pi );


% New A. Moskalik Method - Heywood Paper -- need table entries for 10
% cylinders! Need to do something for 2 cylinder engines, too...

%TODO: Enable distinction between I6 and V6, architecture changes in
%general (using configuration property)

% I6 factors = 40.41 & 2.29

%               1    2      I3      I4      I5      V6      7       V8      9       V10
FMEP_factors = [nan, nan,   41.60,  41.02,  40.65,   41.38,  nan,    40.97,  nan,    40.70;
                nan, nan,   2.82,   2.55,   2.39,    2.34,   nan,    2.21,   nan     2.13];
input_engine_FMEP_kPa =		[1,engine_in.num_cylinders/ engine_in.displacement_L]	* FMEP_factors( :, max(3, min(engine_in.num_cylinders, 8)));
output_engine_FMEP_kPa =	[1,engine_out.num_cylinders/ engine_out.displacement_L] * FMEP_factors( :, max(3, min(engine_in.num_cylinders, 8)));

scaling_friction_adjust_Nm =  scaling_friction_adjust * (output_engine_FMEP_kPa - input_engine_FMEP_kPa) * engine_out.displacement_L ./ (4 * pi );

%% Other Friciton Adjustment

total_friction_adjust_Nm = scaling_friction_adjust_Nm + input_friction_adjust_Nm + input_friction_adjust_bar * 100 * engine_out.displacement_L ./ (4 * pi );

%% Calculate BSFC adjustment for heat loss due to surface to volume effect

if scaling_sv_ratio_adjust
      
    % A Moskalik - makes it reversible
    % Original
    heat_loss_adjust =  (cyl_volume_scale_factor ^-(1/11.5));
    
    % Add Speed Rolloff
    heat_loss_adjust =  1+ (heat_loss_adjust -1 ) .* ( 1250 .* unit_convert.rpm2radps ./ max(speed_mesh_radps,100) );
        
else
    heat_loss_adjust = 1.0;
    
end


%% Compute knock sensitivity factor for scaling or fuel adjustment

if ( scaling_knock_adjust && cyl_volume_scale_factor > 1 ) || ( ~isempty(new_fuel) && fuel_octane_adjust )

    knk_adjust_pwr = engine_in.fuel_map_gps ./ interp1( min_consumption_power_W, min_consumption_fuel_gps, brake_torque_mesh_Nm .* speed_mesh_radps, 'pchip');
    speed_mesh_limited_radps = max(speed_mesh_radps, 20);
    
    
    fuel_per_stroke = engine_in.fuel_map_gps ./ speed_mesh_limited_radps;   
    fuel_per_stroke(above_wot) = nan;

    [min_fuel_per_stroke, min_fuel_per_stroke_idx] = min( fuel_per_stroke, [], 2);

    off_map = isnan(min_fuel_per_stroke);
    
    fuel_per_stroke = max(0, engine_in.fuel_map_gps) ./ speed_mesh_limited_radps;  
    [min_fuel_per_stroke_unlimited, min_fuel_per_stroke_unlimited_idx] = min( fuel_per_stroke, [], 2);
    
    min_fuel_per_stroke(off_map) = min_fuel_per_stroke_unlimited(off_map);
    min_fuel_per_stroke_idx(off_map) = min_fuel_per_stroke_unlimited_idx(off_map);
    
    knk_adjust_fps = fuel_per_stroke ./ min_fuel_per_stroke;

    knk_adjust =  min(knk_adjust_pwr , knk_adjust_fps);   
    knk_adjust( brake_torque_mesh_Nm <  interp1( min_consumption_speed_radps, min_consumption_torque_Nm, speed_mesh_radps) ) = 1.0;
    knk_adjust( speed_mesh_radps > speed_mesh_radps(1,min_fuel_per_stroke_idx)' ) = 1.0;
    knk_adjust( brake_torque_mesh_Nm < 0.3 * engine_in.max_torque_Nm ) = 1.0;

    knk_adjust = 1 + cummax( (knk_adjust - 1)  ./ (ind_torque_mesh_Nm .* speed_mesh_limited_radps) ) .* ind_torque_mesh_Nm .* speed_mesh_limited_radps;

    % Factor for scaling
    scaling_knk_adjust_factor =  cyl_volume_scale_factor / 2.0 * ( scaling_knock_adjust && cyl_volume_scale_factor > 1);
    
    % Factor for fuel
    
    if ~isempty(new_fuel) && fuel_octane_adjust
       
        octane_change = nan;
        octane_params = {'motor_octane_number','anti_knock_index','research_octane_number'} ;
        octane_param_idx = 0;
        
        while isnan(octane_change) && octane_param_idx < length(octane_params)
            octane_param_idx = octane_param_idx+1;
            octane_change = new_fuel.(octane_params{octane_param_idx}) - engine_in.fuel.(octane_params{octane_param_idx});          
            if isempty(octane_change)
                octane_change = nan;
            end
        end
        
        if isnan(octane_change)
            warning('Missing fuel octane data, unable to adjust for differences in fuel octane');
            octane_knk_adjust_factor = 0.0;
        elseif octane_change > 0           
            octane_knk_adjust_factor = 0.0;
        else           
            octane_knk_adjust_factor = -octane_change/4;
        end
        
    else
        octane_knk_adjust_factor = 0.0;        
    end
        
    % Compute total adjustment
    knk_adjust = knk_adjust .^ (scaling_knk_adjust_factor + octane_knk_adjust_factor);
        
    % Compute contour for maximum adjustment (limit range of operation if
    % fuel adjustment too large)
    knk_cdata = contourc(engine_in.fuel_map_speed_radps, engine_in.fuel_map_torque_Nm, knk_adjust,[knk_max_adjust,knk_max_adjust]);
    
    if ~isempty(knk_cdata)
        knk_limit_speed_radps = knk_cdata(1,2:end);
        knk_limit_torque_Nm = knk_cdata(2,2:end);
        
        
        %combine speeds
        combined_speeds_radps = unique([knk_limit_speed_radps(:);engine_in.full_throttle_speed_radps(:)]);
        
        knk_limit_torque_Nm = interp1( knk_limit_speed_radps, knk_limit_torque_Nm,combined_speeds_radps,'linear','extrap');
        orig_full_torque_Nm = interp1( engine_in.full_throttle_speed_radps, engine_in.full_throttle_torque_Nm,combined_speeds_radps,'linear','extrap');
        
        combined_torque_Nm = min(knk_limit_torque_Nm , orig_full_torque_Nm)/2 + orig_full_torque_Nm/2;      
        
        seg_slope = diff( combined_torque_Nm ) ./ diff(combined_speeds_radps);
        same = abs(diff(seg_slope)) < 1e-9* max(combined_torque_Nm) ;
        keep = [true;~same(:);true];
        
        engine_out.full_throttle_speed_radps = combined_speeds_radps(keep);
        engine_out.full_throttle_torque_Nm = max(combined_torque_Nm(keep), 0); 
        
    end

else
    knk_adjust = 1.0;
    
end
    

%%

if ~isempty(new_fuel)     
    fuel_energy_adjust = engine_in.fuel.energy_density_MJpkg / new_fuel.energy_density_MJpkg;
    engine_out.fuel = new_fuel;    
else    
    fuel_energy_adjust = 1.0;    
end


%% Scale Torque and adjust for friction
engine_out.full_throttle_torque_Nm			= engine_out.full_throttle_torque_Nm  .* (scale_factor) - total_friction_adjust_Nm; %for torque curve
engine_out.closed_throttle_torque_Nm		= engine_in.closed_throttle_torque_Nm  .* (scale_factor) - total_friction_adjust_Nm; %for torque curve


if ~isequal(engine_in.naturally_aspirated_torque_Nm , engine_in.full_throttle_torque_Nm ) || ~isequal(engine_out.naturally_aspirated_torque_Nm , engine_out.full_throttle_torque_Nm )
    % Actual Data (not auto calc) need to modify
    engine_out.naturally_aspirated_speed_radps	= engine_in.naturally_aspirated_speed_radps;
    engine_out.naturally_aspirated_torque_Nm	= engine_in.naturally_aspirated_torque_Nm  .* (scale_factor) - total_friction_adjust_Nm; %for torque curve
end

engine_out.fuel_map_torque_Nm				= engine_in.fuel_map_torque_Nm .* (scale_factor) - total_friction_adjust_Nm;   %for BSFC map
engine_out.fuel_map_gps						= engine_in.fuel_map_gps .*(scale_factor).* heat_loss_adjust .* knk_adjust .* fuel_energy_adjust;
engine_out.deac_fuel_map_gps                = engine_in.deac_fuel_map_gps .*(scale_factor).* heat_loss_adjust .* knk_adjust .* fuel_energy_adjust;


if ~isempty(limit_torque)
    engine_out.full_throttle_torque_Nm = min(engine_out.full_throttle_torque_Nm, limit_torque );
end

if ~isempty(limit_bmep)
    limit_torque = limit_bmep * engine_out.displacement_L * 100 / (4*pi);
    engine_out.full_throttle_torque_Nm = min(engine_out.full_throttle_torque_Nm, limit_torque );
end

% Change refrenced mesh to the scaled engine
[speed_mesh_radps, brake_torque_mesh_Nm] = meshgrid( engine_out.fuel_map_speed_radps, engine_out.fuel_map_torque_Nm);
ind_torque_mesh_Nm = brake_torque_mesh_Nm - interp1( engine_in.closed_throttle_speed_radps, engine_in.closed_throttle_torque_Nm, speed_mesh_radps,'linear','extrap');


if rem_deac
    
    engine_out.deac_strategy = [];
    engine_out.deac_fuel_map_gps = [];
    
end

if add_deacC || add_deacD
	

        
    if ~isempty( deac_num_cylinders )
        % Know # of cylinders - nothing to do
        engine_out.deac_num_cylinders = deac_num_cylinders;
    elseif add_deacC
        %Assume All cylinders
        engine_out.deac_num_cylinders = engine_out.num_cylinders;
    else
        % Assume 1/2 as default
        engine_out.deac_num_cylinders = ceil(engine_out.num_cylinders/2);
    end
	
    if strcmpi(deac_method,'reduction curve')
    
        if isempty( deac_max_reduction )
            deac_max_reduction = interp1( [3,4,6,8,100], [0.332, 0.382, 0.482, 0.582, 0.582], engine_out.num_cylinders, 'nearest');
        end

        if isempty( deac_reduction_curve ) % Default reduction curve		
            deac_reduction_curve.bmep_bar =            12* [ -1.00,	0.00	0.05	0.10	0.15	0.20	0.25	0.30	0.35	0.40	0.45	0.50	0.51];
            deac_reduction_curve.fuel_reduction_norm = [	1.0000	1.000	0.7480	0.5457	0.3862	0.2632	0.1710	0.1042	0.0580	0.0281	0.0104	0.000	nan] * deac_max_reduction;	
        end

        
        bmep_mesh_bar = brake_torque_mesh_Nm .*( 4 * pi / (engine_out.displacement_L /1000)*1e-5 );
        deac_fuel_reduction_norm = max(0.0, interp1( deac_reduction_curve.bmep_bar, deac_reduction_curve.fuel_reduction_norm, bmep_mesh_bar, 'linear','extrap'),'includenan');
        

        if add_deacD
            %Discrete Deac Limits maximum potential to 
            deacD_max_reduction = 0.5* engine_out.deac_num_cylinders /engine_out.num_cylinders*deac_max_reduction;
            deac_fuel_reduction_norm = min( deacD_max_reduction , deac_fuel_reduction_norm, 'includenan');            
            deac_fuel_reduction_norm( bmep_mesh_bar > max(deac_reduction_curve.bmep_bar) * (1- engine_out.deac_num_cylinders /engine_out.num_cylinders )) = nan;
        end

        
        
        deac_fuel_reduction_norm( bmep_mesh_bar > max(deac_reduction_curve.bmep_bar)) = nan;
        engine_out.deac_fuel_map_gps = engine_out.fuel_map_gps .* (1- deac_fuel_reduction_norm);
                
    else
        
        if add_deacD
            deac_props = engine_out.deac_num_cylinders / engine_out.num_cylinders;
        else
            deac_props = (1: 0.5: engine_out.deac_num_cylinders) / engine_out.num_cylinders;
        end
        
        engine_out.deac_fuel_map_gps = nan(size(engine_out.fuel_map_gps));
        
        for i = 1:length(deac_props)
        
            deac_load_limit = 0.8;
            fric_trq_Nm = interp1(engine_out.closed_throttle_speed_radps, engine_out.closed_throttle_torque_Nm, engine_out.fuel_map_speed_radps);       
            lu_trq_Nm = (brake_torque_mesh_Nm - fric_trq_Nm')/deac_props(i) + fric_trq_Nm';

            deac_fuel_gps = interp2( engine_out.fuel_map_speed_radps, engine_out.fuel_map_torque_Nm, engine_out.fuel_map_gps, speed_mesh_radps, lu_trq_Nm)*deac_props(i);        
            deac_max_trq_Nm = (deac_load_limit * interp1(engine_out.full_throttle_speed_radps, engine_out.full_throttle_torque_Nm, engine_out.fuel_map_speed_radps)- fric_trq_Nm)*deac_props(i)+fric_trq_Nm;

            deac_fuel_gps(engine_out.fuel_map_torque_Nm * ones(1,length(engine_out.fuel_map_speed_radps)) > ones(length(engine_out.fuel_map_torque_Nm), 1) *deac_max_trq_Nm')=nan;
        
            engine_out.deac_fuel_map_gps = min(engine_out.deac_fuel_map_gps, deac_fuel_gps);
        end
        
    end
    
	% Apply speed limits
	engine_out.deac_fuel_map_gps( speed_mesh_radps < deac_min_speed_radps) = nan;
	engine_out.deac_fuel_map_gps( speed_mesh_radps > deac_max_speed_radps) = nan;	
		    
    % set deac strategy if missing or if wrong type
    engine_out.deac_strategy.variant = 'dynamic lookup';
    engine_out.deac_strategy.enable_norm = class_REVS_dynamic_lookup;
    engine_out.deac_strategy.enable_norm.axis_1.signal =  'gear_num';
    engine_out.deac_strategy.enable_norm.axis_1.breakpoints = [ 1, 2, 3 ];
    engine_out.deac_strategy.enable_norm.axis_2.signal =  'eng_spd_radps';
    engine_out.deac_strategy.enable_norm.axis_2.breakpoints = [ deac_min_speed_radps - [1,0], deac_max_speed_radps + [0,1] ];
    engine_out.deac_strategy.enable_norm.axis_3.signal =  'cycle_pos_secs';
    engine_out.deac_strategy.enable_norm.axis_3.breakpoints =  deac_activation_time + [0, 1];
    engine_out.deac_strategy.enable_norm.table(:,:,1) = [   0.0, 0.0, 0.0, 0.0;
                                                            0.0, 0.0, 0.0, 0.0;
                                                            0.0, 0.0, 0.0, 0.0];
    engine_out.deac_strategy.enable_norm.table(:,:,2) = [   0.00, 0.00, 0.00, 0.00;
                                                            0.00, 0.85, 0.85, 0.00;
                                                            0.00, 1.00, 1.00, 0.00];
    % Default to interpolated deac in transition (activation delay & gear)
    engine_out.deac_strategy.output_digital_bool =  false;  
    engine_out.deac_strategy.enable_threshold  = 0;
    engine_out.deac_strategy.disable_threshold = 0;

	
	% Deac C also theoretically eliminates the need for DFCO
	% May need a more complicated logic to account for cases where DFCO may still be needed (cylinder deac unavailable)
	if add_deacC && engine_out.deac_num_cylinders >= engine_out.num_cylinders
		engine_out.DFCO_refuel_multiplier = max( engine_out.DFCO_refuel_multiplier, 1.0);	
	end
end

%% Ricardo modifier maps are from "RD07-80602-GD012_EPA_InputData-task 6c-part2-v4-file-1of-2.pdf"
%% add or remove GDI effectiveness
if mod_GDI ~= 0
% see REVS_modify_engine() for original implementation
	map_data = [2000.76161500000,6.02555900000000,5;2000.76161500000,5.78274800000000,5;2000.76161500000,5.57827500000000,5;2000.76161500000,5.36102200000000,5;2000.76161500000,5.07987200000000,5;2000.76161500000,5.23322700000000,5;1496.95354200000,6.02555900000000,4.50000000000000;1634.04417400000,6.15335500000000,4.50000000000000;1723.15308500000,6.23003200000000,4.50000000000000;1808.83473000000,6.30670900000000,4.50000000000000;1884.23457700000,6.39616600000000,4.50000000000000;2000.76161500000,6.51118200000000,4.50000000000000;2124.14318400000,6.42172500000000,4.50000000000000;2292.07920800000,6.29393000000000,4.50000000000000;2418.88804300000,6.23003200000000,4.50000000000000;2545.69687700000,6.12779600000000,4.50000000000000;2627.95125700000,6.06389800000000,4.50000000000000;2713.63290200000,6.01278000000000,4.50000000000000;2795.88728100000,5.82108600000000,4.50000000000000;2881.56892600000,5.64217300000000,4.50000000000000;2963.82330500000,5.45047900000000,4.50000000000000;3004.95049500000,5.36102200000000,4.50000000000000;3056.35948200000,5.06709300000000,4.50000000000000;3056.35948200000,4.77316300000000,4.50000000000000;3063.21401400000,4.50479200000000,4.50000000000000;3066.64128000000,4.24920100000000,4.50000000000000;3066.64128000000,3.96805100000000,4.50000000000000;3046.07768500000,3.69968100000000,4.50000000000000;3022.08682400000,3.35463300000000,4.50000000000000;3011.80502700000,3.12460100000000,4.50000000000000;2998.09596300000,3.03514400000000,4.50000000000000;2912.41431800000,3.07348200000000,4.50000000000000;2826.73267300000,3.12460100000000,4.50000000000000;2703.35110400000,3.17571900000000,4.50000000000000;2576.54227000000,3.23961700000000,4.50000000000000;2453.16070100000,3.30351400000000,4.50000000000000;2343.48819500000,3.34185300000000,4.50000000000000;2244.09748700000,3.40575100000000,4.50000000000000;2158.41584200000,3.45686900000000,4.50000000000000;2079.58872800000,3.48242800000000,4.50000000000000;1993.90708300000,3.52076700000000,4.50000000000000;2213.25209400000,6.37060700000000,4.50000000000000;1911.65270400000,3.63578300000000,4.50000000000000;1825.97105900000,3.76357800000000,4.50000000000000;1743.71667900000,3.85303500000000,4.50000000000000;1661.46230000000,3.96805100000000,4.50000000000000;1623.76237600000,4.01916900000000,4.50000000000000;1592.91698400000,4.27476000000000,4.50000000000000;1562.07159200000,4.50479200000000,4.50000000000000;1534.65346500000,4.74760400000000,4.50000000000000;1510.66260500000,4.97763600000000,4.50000000000000;1500.38080700000,5.10543100000000,4.50000000000000;1500.38080700000,5.33546300000000,4.50000000000000;1500.38080700000,5.56549500000000,4.50000000000000;1500.38080700000,5.79552700000000,4.50000000000000;1551.78979400000,6.06389800000000,4.50000000000000;996.572734000000,6.01278000000000,4;996.572734000000,5.70607000000000,4;996.572734000000,5.33546300000000,4;996.572734000000,4.97763600000000,4;996.572734000000,4.58147000000000,4;996.572734000000,4.26198100000000,4;996.572734000000,4.00639000000000,4;1082.25437900000,6.11501600000000,4;1167.93602400000,6.17891400000000,4;1250.19040400000,6.26837100000000,4;1335.87204900000,6.34504800000000,4;1459.25361800000,6.46006400000000,4;1544.93526300000,6.54952100000000,4;1627.18964200000,6.62619800000000,4;1712.87128700000,6.71565500000000,4;1795.12566600000,6.79233200000000,4;1877.38004600000,6.88178900000000,4;1959.63442500000,6.95846600000000,4;2004.18888000000,7.00958500000000,4;2083.01599400000,7.00958500000000,4;2168.69763900000,7.00958500000000,4;2254.37928400000,7.00958500000000,4;2340.06092900000,7.00958500000000,4;2422.31530800000,7.00958500000000,4;2511.42421900000,7.00958500000000,4;2600.53313000000,7.00958500000000,4;2672.50571200000,7.00958500000000,4;2758.18735700000,7.00958500000000,4;2840.44173600000,7.00958500000000,4;2919.26885000000,7.00958500000000,4;3004.95049500000,7.00958500000000,4;3049.50495000000,6.76677300000000,4;3087.20487400000,6.53674100000000,4;3135.18659600000,6.30670900000000,4;3176.31378500000,6.07667700000000,4;3210.58644300000,5.83386600000000,4;3248.28636700000,5.59105400000000,4;3282.55902500000,5.36102200000000,4;3316.83168300000,5.13099000000000,4;3333.96801200000,5.04153400000000,4;3340.82254400000,4.82428100000000,4;3340.82254400000,4.65814700000000,4;3351.10434100000,4.49201300000000,4;3351.10434100000,4.30031900000000,4;3357.95887300000,4.17252400000000,4;3357.95887300000,3.94249200000000,4;3337.39527800000,3.69968100000000,4;3320.25894900000,3.48242800000000,4;3313.40441700000,3.23961700000000,4;3303.12262000000,3.02236400000000,4;3207.15917700000,2.88178900000000,4;3121.47753200000,2.74121400000000,4;3025.51409000000,2.56230000000000,4;2915.84158400000,2.56230000000000,4;2830.15993900000,2.58785900000000,4;2747.90556000000,2.58785900000000,4;2662.22391500000,2.61341900000000,4;2538.84234600000,2.63897800000000,4;2412.03351100000,2.66453700000000,4;2288.65194200000,2.69009600000000,4;2202.97029700000,2.71565500000000,4;2079.58872800000,2.74121400000000,4;1993.90708300000,2.76677300000000,4;1911.65270400000,2.84345000000000,4;1825.97105900000,2.90734800000000,4;1740.28941400000,2.97124600000000,4;1702.58949000000,2.99680500000000,4;1661.46230000000,3.04792300000000,4;1575.78065500000,3.16293900000000,4;1493.52627600000,3.27795500000000,4;1411.27189600000,3.40575100000000,4;1325.59025100000,3.54632600000000,4;1202.20868200000,3.71246000000000,4;1116.52703700000,3.85303500000000,4;1030.84539200000,3.95527200000000,4;1000,6.83067100000000,3.50000000000000;1085.68164500000,6.92012800000000,3.50000000000000;1167.93602400000,6.99680500000000,3.50000000000000;1253.61766900000,7.04792300000000,3.50000000000000;1335.87204900000,7.08626200000000,3.50000000000000;1418.12642800000,7.15016000000000,3.50000000000000;1503.80807300000,7.20127800000000,3.50000000000000;1586.06245200000,7.25239600000000,3.50000000000000;1668.31683200000,7.30351400000000,3.50000000000000;1750.57121100000,7.34185300000000,3.50000000000000;1836.25285600000,7.40575100000000,3.50000000000000;1918.50723500000,7.45686900000000,3.50000000000000;2004.18888000000,7.50798700000000,3.50000000000000;2100.15232300000,7.50798700000000,3.50000000000000;2168.69763900000,7.50798700000000,3.50000000000000;2254.37928400000,7.50798700000000,3.50000000000000;2336.63366300000,7.50798700000000,3.50000000000000;2422.31530800000,7.50798700000000,3.50000000000000;2507.99695400000,7.50798700000000,3.50000000000000;2597.10586400000,7.50798700000000,3.50000000000000;2717.06016800000,7.50798700000000,3.50000000000000;2802.74181300000,7.50798700000000,3.50000000000000;2884.99619200000,7.50798700000000,3.50000000000000;2953.54150800000,7.50798700000000,3.50000000000000;3004.95049500000,7.50798700000000,3.50000000000000;3087.20487400000,7.38019200000000,3.50000000000000;3169.45925400000,7.25239600000000,3.50000000000000;3248.28636700000,7.11182100000000,3.50000000000000;3296.26808800000,7.04792300000000,3.50000000000000;3330.54074600000,6.99680500000000,3.50000000000000;3364.81340400000,6.74121400000000,3.50000000000000;3402.51332800000,6.49840300000000,3.50000000000000;3440.21325200000,6.25559100000000,3.50000000000000;3488.19497300000,5.91054300000000,3.50000000000000;3525.89489700000,5.66773200000000,3.50000000000000;3560.16755500000,5.43770000000000,3.50000000000000;3594.44021300000,5.19488800000000,3.50000000000000;3618.43107400000,5.04153400000000,3.50000000000000;3625.28560500000,4.73482400000000,3.50000000000000;3638.99466900000,4.37699700000000,3.50000000000000;3642.42193500000,4.15974400000000,3.50000000000000;3645.84920000000,3.96805100000000,3.50000000000000;3632.14013700000,3.68690100000000,3.50000000000000;3615.00380800000,3.45686900000000,3.50000000000000;3601.29474500000,3.21405800000000,3.50000000000000;3591.01294700000,3.02236400000000,3.50000000000000;3488.19497300000,2.85623000000000,3.50000000000000;3436.78598600000,2.76677300000000,3.50000000000000;3385.37699900000,2.69009600000000,3.50000000000000;3347.67707500000,2.61341900000000,3.50000000000000;3282.55902500000,2.52396200000000,3.50000000000000;3214.01370900000,2.42172500000000,3.50000000000000;3111.19573500000,2.25559100000000,3.50000000000000;3025.51409000000,2.11501600000000,3.50000000000000;2905.55978700000,2.11501600000000,3.50000000000000;2778.75095200000,2.11501600000000,3.50000000000000;2737.62376200000,2.12779600000000,3.50000000000000;2837.01447100000,2.11501600000000,3.50000000000000;2655.36938300000,2.12779600000000,3.50000000000000;2559.40594100000,2.12779600000000,3.50000000000000;2466.86976400000,2.12779600000000,3.50000000000000;2367.47905600000,2.12779600000000,3.50000000000000;2278.37014500000,2.12779600000000,3.50000000000000;2254.37928400000,2.12779600000000,3.50000000000000;2178.97943600000,2.14057500000000,3.50000000000000;2103.57958900000,2.14057500000000,3.50000000000000;2007.61614600000,2.14057500000000,3.50000000000000;1932.21629900000,2.19169300000000,3.50000000000000;1856.81645100000,2.23003200000000,3.50000000000000;1774.56207200000,2.29393000000000,3.50000000000000;1678.59862900000,2.37060700000000,3.50000000000000;1613.48057900000,2.40894600000000,3.50000000000000;1558.64432600000,2.44728400000000,3.50000000000000;1486.67174400000,2.49840300000000,3.50000000000000;1428.40822500000,2.53674100000000,3.50000000000000;1370.14470700000,2.57508000000000,3.50000000000000;1305.02665700000,2.62619800000000,3.50000000000000;1233.05407500000,2.67731600000000,3.50000000000000;1161.08149300000,2.72843500000000,3.50000000000000;1102.81797400000,2.76677300000000,3.50000000000000;1054.83625300000,2.80511200000000,3.50000000000000;1013.70906300000,2.83067100000000,3.50000000000000;965.727342000000,3.02236400000000,3.50000000000000;941.736481000000,3.25239600000000,3.50000000000000;921.172887000000,3.49520800000000,3.50000000000000;900.609292000000,3.72524000000000,3.50000000000000;876.618431000000,3.98083100000000,3.50000000000000;876.618431000000,4.21086300000000,3.50000000000000;873.191165000000,4.50479200000000,3.50000000000000;873.191165000000,4.79872200000000,3.50000000000000;873.191165000000,5.01597400000000,3.50000000000000;873.191165000000,5.24600600000000,3.50000000000000;880.045697000000,5.46325900000000,3.50000000000000;880.045697000000,5.74440900000000,3.50000000000000;890.327494000000,5.85942500000000,3.50000000000000;890.327494000000,6.02555900000000,3.50000000000000;917.745621000000,6.20447300000000,3.50000000000000;952.018279000000,6.44728400000000,3.50000000000000;972.581874000000,6.61341900000000,3.50000000000000;907.463823000000,6.99680500000000,3;952.018279000000,7.07348200000000,3;1000,7.15016000000000,3;1082.25437900000,7.22683700000000,3;1167.93602400000,7.30351400000000,3;1250.19040400000,7.38019200000000,3;1291.31759300000,7.41853000000000,3;1376.99923800000,7.48242800000000,3;1459.25361800000,7.55910500000000,3;1544.93526300000,7.62300300000000,3;1630.61690800000,7.69968100000000,3;1712.87128700000,7.76357800000000,3;1798.55293200000,7.84025600000000,3;1880.80731200000,7.90415300000000,3;1963.06169100000,7.98083100000000,3;2004.18888000000,8.00639000000000,3;2083.01599400000,8.00639000000000,3;2168.69763900000,8.00639000000000,3;2254.37928400000,8.00639000000000,3;2374.33358700000,8.00639000000000,3;2490.86062500000,8.00639000000000,3;2586.82406700000,8.00639000000000,3;2672.50571200000,8.00639000000000,3;2751.33282600000,8.00639000000000,3;2837.01447100000,8.00639000000000,3;2926.12338200000,8.00639000000000,3;3004.95049500000,8.00639000000000,3;3083.77760900000,7.89137400000000,3;3172.88651900000,7.75079900000000,3;3255.14089900000,7.63578300000000,3;3340.82254400000,7.48242800000000,3;3419.64965700000,7.38019200000000,3;3505.33130200000,7.23961700000000,3;3591.01294700000,7.11182100000000,3;3669.84006100000,6.97124600000000,3;3690.40365600000,6.74121400000000,3;3714.39451600000,6.49840300000000,3;3741.81264300000,6.26837100000000,3;3745.23990900000,6.15335500000000,3;3765.80350300000,6.03833900000000,3;3776.08530100000,5.91054300000000,3;3806.93069300000,5.68051100000000,3;3841.20335100000,5.45047900000000,3;3851.48514900000,5.32268400000000,3;3865.19421200000,5.20766800000000,3;3882.33054100000,5.09265200000000,3;3892.61233800000,4.96485600000000,3;3902.89413600000,4.84984000000000,3;3906.32140100000,4.73482400000000,3;3906.32140100000,4.61980800000000,3;3916.60319900000,4.49201300000000,3;3923.45773000000,4.33865800000000,3;3923.45773000000,4.27476000000000,3;3930.31226200000,4.14696500000000,3;3940.59405900000,4.03194900000000,3;3930.31226200000,3.90415300000000,3;3930.31226200000,3.80191700000000,3;3916.60319900000,3.55910500000000,3;3902.89413600000,3.31629400000000,3;3889.18507200000,3.21405800000000,3;3892.61233800000,3.08626200000000,3;3865.19421200000,2.99680500000000,3;3779.51256700000,2.85623000000000,3;3697.25818700000,2.71565500000000,3;3618.43107400000,2.57508000000000,3;3532.74942900000,2.42172500000000,3;3447.06778400000,2.28115000000000,3;3361.38613900000,2.12779600000000,3;3282.55902500000,2.01278000000000,3;3196.87738000000,1.92332300000000,3;3111.19573500000,1.84664500000000,3;3032.36862100000,1.75718800000000,3;2987.81416600000,1.73162900000000,3;2946.68697600000,1.73162900000000,3;2902.13252100000,1.74440900000000,3;2823.30540700000,1.74440900000000,3;2778.75095200000,1.74440900000000,3;2747.90556000000,1.76996800000000,3;2614.24219300000,1.76996800000000,3;2528.56054800000,1.76996800000000,3;2487.43335900000,1.79552700000000,3;2401.75171400000,1.79552700000000,3;2319.49733400000,1.80830700000000,3;2278.37014500000,1.82108600000000,3;2192.68850000000,1.82108600000000,3;2113.86138600000,1.83386600000000,3;2069.30693100000,1.84664500000000,3;1997.33434900000,1.84664500000000,3;1945.92536200000,1.84664500000000,3;1901.37090600000,1.87220400000000,3;1819.11652700000,1.87220400000000,3;1777.98933700000,1.89776400000000,3;1654.60776800000,1.89776400000000,3;1712.87128700000,1.89776400000000,3;1610.05331300000,1.91054300000000,3;1524.37166800000,1.91054300000000,3;1483.24447800000,1.92332300000000,3;1400.99009900000,1.92332300000000,3;1359.86290900000,1.93610200000000,3;1315.30845400000,1.96166100000000,3;1233.05407500000,1.96166100000000,3;1191.92688500000,1.98722000000000,3;1106.24524000000,1.98722000000000,3;1027.41812600000,2.01278000000000,3;1000,2,3;965.727342000000,2.10223600000000,3;941.736481000000,2.28115000000000,3;900.609292000000,2.53674100000000,3;852.627570000000,2.75399400000000,3;808.073115000000,2.98402600000000,3;790.936786000000,3.11182100000000,3;784.082254000000,3.22683700000000,3;780.654989000000,3.35463300000000,3;773.800457000000,3.46964900000000,3;766.945925000000,3.58466500000000,3;749.809596000000,3.82747600000000,3;742.955065000000,3.94249200000000,3;742.955065000000,4.17252400000000,3;742.955065000000,4.41533500000000,3;742.955065000000,4.64536700000000,3;742.955065000000,4.87539900000000,3;742.955065000000,5.11821100000000,3;749.809596000000,5.36102200000000,3;756.664128000000,5.59105400000000,3;766.945925000000,5.82108600000000,3;784.082254000000,6.06389800000000,3;814.927647000000,6.29393000000000,3;842.345773000000,6.52396200000000,3;876.618431000000,6.75399400000000,3;890.327494000000,6.88178900000000,3;806.839940000000,7.02258800000000,2.50000000000000;889.863052000000,7.15127700000000,2.50000000000000;931.374608000000,7.25422800000000,2.50000000000000;1004.01983000000,7.37004800000000,2.50000000000000;1083.58364600000,7.43439300000000,2.50000000000000;1170.06605400000,7.51160700000000,2.50000000000000;1253.08916600000,7.57595100000000,2.50000000000000;1332.65298100000,7.64029600000000,2.50000000000000;1415.67609300000,7.71750900000000,2.50000000000000;1502.15850100000,7.79472300000000,2.50000000000000;1586,7.87193600000000,2.50000000000000;1668.20472500000,7.93628100000000,2.50000000000000;1751.22783700000,8.00062600000000,2.50000000000000;1837.71024500000,8.07783900000000,2.50000000000000;1917.27406000000,8.16792200000000,2.50000000000000;2003.75646800000,8.23226600000000,2.50000000000000;2086.77958000000,8.25800400000000,2.50000000000000;2045.26802400000,8.24226600000000,2.50000000000000;2169.80269200000,8.28374200000000,2.50000000000000;2252.82580400000,8.31000000000000,2.50000000000000;2335.84891600000,8.33521700000000,2.50000000000000;2418.87202700000,8.36095500000000,2.50000000000000;2505.35443600000,8.38669300000000,2.50000000000000;2588.37754700000,8.40466930000000,2.50000000000000;2671.40065900000,8.41243100000000,2.50000000000000;2754.42377100000,8.43816900000000,2.50000000000000;2837.44688300000,8.45103800000000,2.50000000000000;2920.46999500000,8.47677600000000,2.50000000000000;3006.95240300000,8.50251300000000,2.50000000000000;3048.46395900000,8.46392200000000,2.50000000000000;3089.97551400000,8.45090100000000,2.50000000000000;3172.99862600000,8.38579800000000,2.50000000000000;3252.56244200000,8.33371500000000,2.50000000000000;3339.04485000000,8.25559100000000,2.50000000000000;3425.52725800000,8.21652900000000,2.50000000000000;3505.09107300000,8.15142600000000,2.50000000000000;3591.57348200000,8.09934300000000,2.50000000000000;3633.08503700000,8.04726000000000,2.50000000000000;3674.59659300000,8.02121900000000,2.50000000000000;3723.02674200000,8.00819800000000,2.50000000000000;3747.24181600000,7.89101200000000,2.50000000000000;3812.96844600000,7.65664000000000,2.50000000000000;3882.15437300000,7.42226800000000,2.50000000000000;3951.34029900000,7.17487500000000,2.50000000000000;4017.06693000000,6.94050300000000,2.50000000000000;4117.38652300000,6.71915100000000,2.50000000000000;4058.57848500000,6.86237900000000,2.50000000000000;4183.11315300000,6.54988200000000,2.50000000000000;4266.13626500000,6.34155200000000,2.50000000000000;4349.15937700000,6.12020000000000,2.50000000000000;4394.13022900000,6.02905500000000,2.50000000000000;4414.88600700000,5.92489000000000,2.50000000000000;4466.77545200000,5.67749700000000,2.50000000000000;4518.66489700000,5.44312500000000,2.50000000000000;4560.17645300000,5.19573200000000,2.50000000000000;4587.85082300000,5.09156700000000,2.50000000000000;4608.60660100000,4.85719500000000,2.50000000000000;4625.90308300000,4.62282200000000,2.50000000000000;4643.19956400000,4.38845000000000,2.50000000000000;4660.49604600000,4.15407800000000,2.50000000000000;4667.41463900000,4.02387100000000,2.50000000000000;4632.82167500000,3.80252000000000,2.50000000000000;4591.31012000000,3.55512700000000,2.50000000000000;4549.79856400000,3.33377500000000,2.50000000000000;4532.50208200000,3.20356900000000,2.50000000000000;4511.74630400000,3.09940300000000,2.50000000000000;4501.36841500000,3.00825800000000,2.50000000000000;4442.56037800000,2.96919600000000,2.50000000000000;4356.07796900000,2.90409300000000,2.50000000000000;4276.51415400000,2.83899000000000,2.50000000000000;4231.54330200000,2.79992800000000,2.50000000000000;4190.03174600000,2.77388600000000,2.50000000000000;4148.52019000000,2.74784500000000,2.50000000000000;4065.49707800000,2.69576200000000,2.50000000000000;4023.98552200000,2.66972100000000,2.50000000000000;3982.47396600000,2.61763800000000,2.50000000000000;3899.45085500000,2.46139000000000,2.50000000000000;3812.96844600000,2.30514200000000,2.50000000000000;3733.40463100000,2.16191500000000,2.50000000000000;3646.92222300000,2.03170800000000,2.50000000000000;3563.89911100000,1.92754200000000,2.50000000000000;3480.87599900000,1.84941800000000,2.50000000000000;3439.36444300000,1.79733600000000,2.50000000000000;3397.85288700000,1.77129400000000,2.50000000000000;3314.82977600000,1.66712900000000,2.50000000000000;3273.31822000000,1.62806700000000,2.50000000000000;3231.80666400000,1.60202500000000,2.50000000000000;3148.78355200000,1.51088100000000,2.50000000000000;3065.76044000000,1.40671500000000,2.50000000000000;2979.27803200000,1.38067400000000,2.50000000000000;2896.25492000000,1.39369500000000,2.50000000000000;2813.23180800000,1.40671500000000,2.50000000000000;2726.74940000000,1.43275700000000,2.50000000000000;2647.18558500000,1.45879800000000,2.50000000000000;2564.16247300000,1.48483900000000,2.50000000000000;2481.13936100000,1.51088100000000,2.50000000000000;2394.65695300000,1.53692200000000,2.50000000000000;2311.63384100000,1.56296300000000,2.50000000000000;2225.15143300000,1.58900500000000,2.50000000000000;2145.58761800000,1.58900500000000,2.50000000000000;2059.10521000000,1.61504600000000,2.50000000000000;1976.08209800000,1.62806700000000,2.50000000000000;1855.00672600000,1.64108700000000,2.50000000000000;1768.52431800000,1.64108700000000,2.50000000000000;1685.50120600000,1.66712900000000,2.50000000000000;1599.01879800000,1.66712900000000,2.50000000000000;1519.45498300000,1.69317000000000,2.50000000000000;1432.97257500000,1.69317000000000,2.50000000000000;1391.46101900000,1.71921200000000,2.50000000000000;1308.43790700000,1.71921200000000,2.50000000000000;1266.92635100000,1.71921200000000,2.50000000000000;1228.87409200000,1.74525300000000,2.50000000000000;1183.90323900000,1.74525300000000,2.50000000000000;1100.88012800000,1.74525300000000,2.50000000000000;1017.85701600000,1.77129400000000,2.50000000000000;976.345460000000,1.79733600000000,2.50000000000000;931.374608000000,1.83639800000000,2.50000000000000;848.351496000000,1.92754200000000,2.50000000000000;806.839940000000,1.97962500000000,2.50000000000000;758.409791000000,2.20097700000000,2.50000000000000;709.979643000000,2.52649300000000,2.50000000000000;668.468087000000,2.77388600000000,2.50000000000000;633.875124000000,3.00825800000000,2.50000000000000;623.497235000000,3.25565100000000,2.50000000000000;623.497235000000,3.49002400000000,2.50000000000000;613.119346000000,3.72439600000000,2.50000000000000;606.200753000000,3.84158200000000,2.50000000000000;606.200753000000,4.07595400000000,2.50000000000000;606.200753000000,4.32334700000000,2.50000000000000;606.200753000000,4.54469800000000,2.50000000000000;606.200753000000,4.77907000000000,2.50000000000000;606.200753000000,5.01344300000000,2.50000000000000;623.497235000000,5.24781500000000,2.50000000000000;633.875124000000,5.49520800000000,2.50000000000000;647.712309000000,5.71655900000000,2.50000000000000;668.468087000000,5.96395200000000,2.50000000000000;689.223865000000,6.19832400000000,2.50000000000000;741.113310000000,6.54988200000000,2.50000000000000;775.706273000000,6.78425500000000,2.50000000000000;709.979643000000,6.31551000000000,2.50000000000000;2117.91324700000,8.54204600000000,2;2079.86098800000,8.54204600000000,2;1993.37857900000,8.48996300000000,2;1910.35546800000,8.41183900000000,2;1827.33235600000,8.33371500000000,2;1744.30924400000,8.24257000000000,2;1661.28613200000,8.16444600000000,2;1578.26302000000,8.07330200000000,2;1533.29216800000,8.02121900000000,2;1453.72835300000,7.95611500000000,2;1367.24594500000,7.89101200000000,2;1280.76353600000,7.81288800000000,2;1201.19972100000,7.74778500000000,2;1118.17660900000,7.66966100000000,2;1073.20575700000,7.65664000000000,2;993.641941000000,7.56549500000000,2;907.159533000000,7.40924700000000,2;865.647977000000,7.33112300000000,2;786.084162000000,7.17487500000000,2;741.113310000000,7.12279200000000,2;699.601754000000,7.01862700000000,2;658.090198000000,6.81029600000000,2;626.956531000000,6.57592400000000,2;588.904271000000,6.32853100000000,2;564.689197000000,6.09415900000000,2;533.555530000000,5.75562100000000,2;512.799752000000,5.52124900000000,2;512.799752000000,2.53951400000000,2;547.392716000000,2.30514200000000,2;533.555530000000,2.42232800000000,2;557.770605000000,2.18795600000000,2;571.607790000000,2.07077000000000,2;599.282160000000,1.97962500000000,2;678.845976000000,1.87546000000000,2;730.735421000000,1.83639800000000,2;806.839940000000,1.74525300000000,2;889.863052000000,1.62806700000000,2;972.886164000000,1.53692200000000,2;1017.85701600000,1.51088100000000,2;1100.88012800000,1.51088100000000,2;1142.39168300000,1.51088100000000,2;1183.90323900000,1.48483900000000,2;1266.92635100000,1.48483900000000,2;1349.94946300000,1.48483900000000,2;1432.97257500000,1.48483900000000,2;1477.94342700000,1.48483900000000,2;1519.45498300000,1.45879800000000,2;1605.93739100000,1.45879800000000,2;1716.63487300000,1.45879800000000,2;1810.03587400000,1.45879800000000,2;1851.54743000000,1.43275700000000,2;1934.57054200000,1.43275700000000,2;2017.59365400000,1.43275700000000,2;2059.10521000000,1.40671500000000,2;2145.58761800000,1.36765300000000,2;2228.61073000000,1.34161200000000,2;2311.63384100000,1.30255000000000,2;2394.65695300000,1.27650900000000,2;2477.68006500000,1.22442600000000,2;2560.70317700000,1.19838400000000,2;2647.18558500000,1.15932200000000,2;2685.23784400000,1.15932200000000,2;2771.72025300000,1.12026000000000,2;2854.74336400000,1.08119800000000,2;2937.76647600000,1.04213600000000,2;2979.27803200000,1.01609500000000,2;3065.76044000000,1.06817800000000,2;3186.83581200000,1.19838400000000,2;3314.82977600000,1.32859100000000,2;3442.82374000000,1.43275700000000,2;3567.35840700000,1.58900500000000,2;3691.89307500000,1.70619100000000,2;3816.42774300000,1.82337700000000,2;3899.45085500000,1.90150100000000,2;3982.47396600000,1.97962500000000,2;4023.98552200000,2.03170800000000,2;4107.00863400000,2.05774900000000,2;4190.03174600000,2.09681100000000,2;4273.05485800000,2.14889400000000,2;4401.04882200000,2.21399700000000,2;4525.58348900000,2.27910100000000,2;4608.60660100000,2.30514200000000,2;4691.62971300000,2.35722500000000,2;4816.16438100000,2.43534900000000,2;4982.21060400000,2.51347300000000,2;5068.69301200000,2.50045200000000,2;5193.22768000000,2.43534900000000,2;5321.22164400000,2.35722500000000,2;5442.29701500000,2.27910100000000,2;5566.83168300000,2.22701800000000,2;5694.82564700000,2.16191500000000,2;5781.30805500000,2.10983200000000,2;5860.87187100000,2.07077000000000,2;5902.38342700000,2.05774900000000,2;5961.19146400000,2.03170800000000,2;6000,2.01868700000000,2;6000,7.99517800000000,2;5923.13920500000,8.07330200000000,2;5836.65679700000,8.17746700000000,2;5712.12212900000,8.28163200000000,2;5584.12816500000,8.41183900000000,2;5504.56434900000,8.48996300000000,2;5376.57038500000,8.62017000000000,2;5293.54727300000,8.69829400000000,2;5169.01260600000,8.82850100000000,2;5041.01864200000,8.94568700000000,2;5002.96638200000,8.99777000000000,2;4836.92015900000,8.99777000000000,2;4681.25182400000,8.99777000000000,2;4501.36841500000,8.99777000000000,2;4290.35133900000,8.99777000000000,2;4124.30511600000,8.99777000000000,2;3958.25889200000,8.99777000000000,2;3792.21266800000,8.99777000000000,2;3626.16644500000,8.99777000000000,2;3498.17248100000,8.99777000000000,2;3328.66696100000,8.99777000000000,2;3148.78355200000,8.99777000000000,2;2996.57451400000,8.99777000000000,2;3038.08607000000,8.99777000000000,2;2955.06295800000,8.97172800000000,2;4964.91412300000,8.99777000000000,2;4944.15834500000,2.50045200000000,2;2830.52829000000,8.89360400000000,2;2747.50517800000,8.88058400000000,2;2661.02277000000,8.82850100000000,2;2619.51121400000,8.80245900000000,2;2536.48810200000,8.76339700000000,2;2456.92428700000,8.71131500000000,2;2370.44187900000,8.68527300000000,2;2287.41876700000,8.64621100000000,2;2200.93635900000,8.59412900000000,2;2159.42480300000,8.56808700000000,2;2915.84158400000,8.95207700000000,2;3080.35034300000,9.00319500000000,2;3121.47753200000,9.00319500000000,2;3207.15917700000,9.00319500000000,2;3248.28636700000,9.00319500000000,2;3289.41355700000,9.00319500000000,2;3375.09520200000,9.00319500000000,2;3433.35872000000,9.00319500000000,2;3539.60396000000,9.00319500000000,2;3580.73115000000,9.00319500000000,2;3669.84006100000,9.00319500000000,2;3731.53084500000,9.00319500000000,2;3844.63061700000,9.00319500000000,2;3889.18507200000,9.00319500000000,2;4019.42117300000,9.00319500000000,2;4067.40289400000,9.00319500000000,2;4166.79360200000,9.00319500000000,2;4204.49352600000,9.00319500000000,2;4249.04798200000,9.00319500000000,2;4372.42955100000,9.00319500000000,2;4427.26580400000,9.00319500000000,2;4547.22010700000,9.00319500000000,2;4595.20182800000,9.00319500000000,2;4629.47448600000,9.00319500000000,2;4718.58339700000,9.00319500000000,2;4756.28332100000,9.00319500000000,2;4790.55597900000,9.00319500000000,2;4879.66489000000,9.00319500000000,2;4924.21934500000,9.00319500000000,2;5085.30083800000,8.90095800000000,2;5123.00076200000,8.87539900000000,2;5212.10967300000,8.78594200000000,2;5256.66412800000,8.73482400000000,2;5332.06397600000,8.67092700000000,2;5421.17288700000,8.58147000000000,2;5462.30007600000,8.54313100000000,2;5544.55445500000,8.45367400000000,2;5630.23610100000,8.37699700000000,2;5671.36329000000,8.33865800000000,2;5753.61766900000,8.24920100000000,2;5801.59939100000,8.19808300000000,2;5880.42650400000,8.12140600000000,2;5962.68088300000,8.04472800000000,2;997.101238000000,7.78684700000000,1.50000000000000;1083.58364600000,7.85195000000000,1.50000000000000;1166.60675800000,7.93007400000000,1.50000000000000;1249.62987000000,7.99517800000000,1.50000000000000;1346.49016700000,8.09934300000000,1.50000000000000;1419.13538900000,8.16444600000000,1.50000000000000;1502.15850100000,8.25559100000000,1.50000000000000;1585.18161300000,8.33371500000000,1.50000000000000;1668.20472500000,8.43788100000000,1.50000000000000;1751.22783700000,8.48996300000000,1.50000000000000;1837.71024500000,8.59412900000000,1.50000000000000;1920.73335700000,8.67225300000000,1.50000000000000;2003.75646800000,8.76339700000000,1.50000000000000;2086.77958000000,8.80245900000000,1.50000000000000;2169.80269200000,8.82850100000000,1.50000000000000;2256.28510000000,8.88058400000000,1.50000000000000;2335.84891600000,8.91964600000000,1.50000000000000;2422.33132400000,8.94568700000000,1.50000000000000;2505.35443600000,8.99777000000000,1.50000000000000;2588.37754700000,9.03683200000000,1.50000000000000;2674.85995500000,9.08891400000000,1.50000000000000;2757.88306700000,9.12797600000000,1.50000000000000;2837.44688300000,9.15401800000000,1.50000000000000;2923.92929100000,9.20610000000000,1.50000000000000;3006.95240300000,9.25818300000000,1.50000000000000;3141.86495900000,9.25818300000000,1.50000000000000;3214.51018200000,9.25818300000000,1.50000000000000;3252.56244200000,9.28422400000000,1.50000000000000;3384.01570200000,9.28422400000000,1.50000000000000;3505.09107300000,9.28422400000000,1.50000000000000;3636.54433400000,9.28422400000000,1.50000000000000;3716.10814900000,9.28422400000000,1.50000000000000;3757.61970500000,9.31026600000000,1.50000000000000;3882.15437300000,9.31026600000000,1.50000000000000;3965.17748500000,9.31026600000000,1.50000000000000;4006.68904100000,9.31026600000000,1.50000000000000;4093.17144900000,9.32328700000000,1.50000000000000;4172.73526400000,9.33630700000000,1.50000000000000;4259.21767200000,9.34932800000000,1.50000000000000;4342.24078400000,9.36234900000000,1.50000000000000;4425.26389600000,9.36234900000000,1.50000000000000;4511.74630400000,9.38839000000000,1.50000000000000;4594.76941600000,9.41443100000000,1.50000000000000;4674.33323100000,9.41443100000000,1.50000000000000;4719.30408400000,9.44047300000000,1.50000000000000;4798.86789900000,9.44047300000000,1.50000000000000;4840.37945500000,9.47953500000000,1.50000000000000;4926.86186300000,9.46651400000000,1.50000000000000;4968.37341900000,9.49255500000000,1.50000000000000;5009.88497500000,9.49255500000000,1.50000000000000;5096.36738300000,9.38839000000000,1.50000000000000;5134.41964300000,9.34932800000000,1.50000000000000;5179.39049500000,9.32328700000000,1.50000000000000;5300.46586600000,9.18005900000000,1.50000000000000;5428.45983000000,9.06287300000000,1.50000000000000;5552.99449800000,8.94568700000000,1.50000000000000;5684.44775800000,8.82850100000000,1.50000000000000;5802.06383300000,8.68527300000000,1.50000000000000;5888.54624100000,8.62017000000000,1.50000000000000;5971.56935300000,8.51600500000000,1.50000000000000;6000,8.48996300000000,1.50000000000000;6000,1.39369500000000,1.50000000000000;5954.27287200000,1.39369500000000,1.50000000000000;5912.76131600000,1.40671500000000,1.50000000000000;5871.24976000000,1.40671500000000,1.50000000000000;5826.27890800000,1.43275700000000,1.50000000000000;5784.76735200000,1.43275700000000,1.50000000000000;5743.25579600000,1.45879800000000,1.50000000000000;5660.23268400000,1.45879800000000,1.50000000000000;5618.72112800000,1.48483900000000,1.50000000000000;5577.20957200000,1.48483900000000,1.50000000000000;5535.69801600000,1.51088100000000,1.50000000000000;5452.67490400000,1.51088100000000,1.50000000000000;5411.16334900000,1.53692200000000,1.50000000000000;5366.19249600000,1.52390100000000,1.50000000000000;5324.68094000000,1.54994300000000,1.50000000000000;5283.16938500000,1.54994300000000,1.50000000000000;5245.11712500000,1.58900500000000,1.50000000000000;5162.09401300000,1.57598400000000,1.50000000000000;5120.58245700000,1.60202500000000,1.50000000000000;5079.07090100000,1.60202500000000,1.50000000000000;5037.55934500000,1.62806700000000,1.50000000000000;4954.53623400000,1.62806700000000,1.50000000000000;4909.56538100000,1.62806700000000,1.50000000000000;4868.05382600000,1.60202500000000,1.50000000000000;4785.03071400000,1.60202500000000,1.50000000000000;4702.00760200000,1.60202500000000,1.50000000000000;4660.49604600000,1.58900500000000,1.50000000000000;4598.22871200000,1.58900500000000,1.50000000000000;4532.50208200000,1.58900500000000,1.50000000000000;4490.99052600000,1.56296300000000,1.50000000000000;4407.96741400000,1.56296300000000,1.50000000000000;4328.40359900000,1.56296300000000,1.50000000000000;4283.43274700000,1.53692200000000,1.50000000000000;4203.86893100000,1.53692200000000,1.50000000000000;4117.38652300000,1.53692200000000,1.50000000000000;4072.41567100000,1.51088100000000,1.50000000000000;3989.39255900000,1.51088100000000,1.50000000000000;3951.34029900000,1.45879800000000,1.50000000000000;3864.85789100000,1.36765300000000,1.50000000000000;3781.83477900000,1.30255000000000,1.50000000000000;3702.27096400000,1.19838400000000,1.50000000000000;3574.27700000000,1.08119800000000,1.50000000000000;3491.25388800000,1.00307400000000,1.50000000000000;3404.77148000000,0.964012000000000,1.50000000000000;3321.74836800000,0.924950000000000,1.50000000000000;3197.21370100000,0.846826000000000,1.50000000000000;3114.19058900000,0.794744000000000,1.50000000000000;3031.16747700000,0.768702000000000,1.50000000000000;2989.65592100000,0.742661000000000,1.50000000000000;2906.63280900000,0.794744000000000,1.50000000000000;2823.60969700000,0.833806000000000,1.50000000000000;2695.61573300000,0.911930000000000,1.50000000000000;2612.59262200000,0.950992000000000,1.50000000000000;2529.56951000000,0.990054000000000,1.50000000000000;2405.03484200000,1.05515700000000,1.50000000000000;2322.01173000000,1.08119800000000,1.50000000000000;2194.01776600000,1.14630200000000,1.50000000000000;2110.99465400000,1.17234300000000,1.50000000000000;2027.97154300000,1.21140500000000,1.50000000000000;1944.94843100000,1.22442600000000,1.50000000000000;1778.90220700000,1.22442600000000,1.50000000000000;1861.92531900000,1.22442600000000,1.50000000000000;1733.93135500000,1.25046700000000,1.50000000000000;1602.47809500000,1.25046700000000,1.50000000000000;1446.80976000000,1.25046700000000,1.50000000000000;1277.30424000000,1.25046700000000,1.50000000000000;1235.79268400000,1.27650900000000,1.50000000000000;1152.76957200000,1.27650900000000,1.50000000000000;1069.74646100000,1.27650900000000,1.50000000000000;1007.47912700000,1.27650900000000,1.50000000000000;945.211793000000,1.31557100000000,1.50000000000000;851.810792000000,1.38067400000000,1.50000000000000;720.357532000000,1.45879800000000,1.50000000000000;644.253013000000,1.52390100000000,1.50000000000000;568.148493000000,1.57598400000000,1.50000000000000;512.799752000000,1.62806700000000,1.50000000000000;505.881160000000,6.53686200000000,1.50000000000000;540.474123000000,6.77123400000000,1.50000000000000;571.607790000000,6.97956500000000,1.50000000000000;665.008791000000,7.14883400000000,1.50000000000000;751.491199000000,7.33112300000000,1.50000000000000;834.514311000000,7.48737100000000,1.50000000000000;917.537422000000,7.63059900000000,1.50000000000000;509.340456000000,7.16185400000000,1;592.363568000000,7.31810200000000,1;678.845976000000,7.46133000000000,1;761.869088000000,7.60455700000000,1;841.432903000000,7.74778500000000,1;924.456015000000,7.87799100000000,1;1000.56053400000,7.99517800000000,1;1093.96153500000,8.09934300000000,1;1173.52535000000,8.16444600000000,1;1260.00775900000,8.25559100000000,1;1343.03087000000,8.33371500000000,1;1426.05398200000,8.42486000000000,1;1509.07709400000,8.48996300000000,1;1592.10020600000,8.59412900000000,1;1671.66402100000,8.65923200000000,1;1761.60572600000,8.75037700000000,1;1841.16954100000,8.82850100000000,1;1927.65194900000,8.91964600000000,1;2003.75646800000,8.99777000000000,1;2097.15746900000,9.03683200000000,1;2180.18058100000,9.08891400000000,1;2263.20369300000,9.11495600000000,1;2342.76750800000,9.15401800000000,1;2432.70921300000,9.20610000000000,1;2515.73232400000,9.24516200000000,1;2595.29614000000,9.27120400000000,1;2678.31925200000,9.32328700000000,1;2764.80166000000,9.36234900000000,1;2847.82477200000,9.41443100000000,1;2930.84788300000,9.44047300000000,1;3010.41169900000,9.49255500000000,1;3100.35340300000,9.49255500000000,1;3141.86495900000,9.51859700000000,1;3228.34736700000,9.51859700000000,1;3349.42273900000,9.51859700000000,1;3394.39359100000,9.54463800000000,1;3473.95740700000,9.54463800000000,1;3598.49207400000,9.54463800000000,1;3681.51518600000,9.55765900000000,1;3723.02674200000,9.55765900000000,1;3809.50915000000,9.55765900000000,1;3851.02070600000,9.55765900000000,1;3892.53226200000,9.58370000000000,1;3979.01467000000,9.57067900000000,1;4017.06693000000,9.58370000000000,1;4100.09004100000,9.62276200000000,1;4186.57244900000,9.66182400000000,1;4269.59556100000,9.70088600000000,1;4394.13022900000,9.75296900000000,1;4477.15334100000,9.79203100000000,1;4605.14730500000,9.83109300000000,1;4733.14126900000,9.88317600000000,1;4850.75734400000,9.93525800000000,1;4937.23975200000,9.98734100000000,1;5002.96638200000,10.0133820000000,1;5148.25682800000,9.85713400000000,1;5310.84375500000,9.70088600000000,1;5438.83771900000,9.55765900000000,1;5566.83168300000,9.44047300000000,1;5687.90705500000,9.29724500000000,1;5815.90101900000,9.18005900000000,1;5898.92413000000,9.08891400000000,1;5940.43568600000,9.06287300000000,1;6000,8.98474900000000,1;6002.70302000000,0.846826000000000,1;5943.89498300000,0.846826000000000,1;5895.46483400000,0.872868000000000,1;5812.44172200000,0.872868000000000,1;5774.38946300000,0.898909000000000,1;5729.41861000000,0.898909000000000,1;5646.39549900000,0.898909000000000,1;5604.88394300000,0.924950000000000,1;5518.40153500000,0.924950000000000,1;5480.34927500000,0.937971000000000,1;5397.32616300000,0.937971000000000,1;5352.35531100000,0.964012000000000,1;5227.82064300000,0.964012000000000,1;5186.30908700000,0.990054000000000,1;5103.28597600000,0.990054000000000,1;5061.77442000000,1.01609500000000,1;4885.35030700000,1.00307400000000,1;4598.22871200000,1.00307400000000,1;4446.01967400000,1.00307400000000,1;4293.81063600000,1.00307400000000,1;4767.73423200000,0.990054000000000,1;4179.65385700000,1.00307400000000,1;4017.06693000000,1.00307400000000,1;3892.53226200000,0.950992000000000,1;3764.53829800000,0.885888000000000,1;3601.95137100000,0.807764000000000,1;3473.95740700000,0.742661000000000,1;3390.93429500000,0.716619000000000,1;3311.37047900000,0.664537000000000,1;3224.88807100000,0.625475000000000,1;3138.40566300000,0.573392000000000,1;3065.76044000000,0.534330000000000,1;2972.35943900000,0.534330000000000,1;2837.44688300000,0.586413000000000,1;2681.77854800000,0.677557000000000,1;2560.70317700000,0.729640000000000,1;2432.70921300000,0.794744000000000,1;2263.20369300000,0.885888000000000,1;2135.20972900000,0.937971000000000,1;2014.13435700000,1.00307400000000,1;1972.62280100000,1.01609500000000,1;1841.16954100000,1.01609500000000,1;1695.87909500000,1.01609500000000,1;1543.67005700000,1.01609500000000,1;1246.17057300000,1.01609500000000,1;983.264053000000,1.00307400000000,1;824.136422000000,1.00307400000000,1;661.549494000000,1.00307400000000,1;505.881160000000,1.00307400000000,1;505.881160000000,7.56549500000000,0.500000000000000;592.363568000000,7.72174300000000,0.500000000000000;675.386680000000,7.86497100000000,0.500000000000000;758.409791000000,8.02121900000000,0.500000000000000;844.892200000000,8.20350800000000,0.500000000000000;924.456015000000,8.35975600000000,0.500000000000000;1007.47912700000,8.50298400000000,0.500000000000000;1093.96153500000,8.59412900000000,0.500000000000000;1173.52535000000,8.65923200000000,0.500000000000000;1260.00775900000,8.75037700000000,0.500000000000000;1343.03087000000,8.82850100000000,0.500000000000000;1471.02483400000,8.95870800000000,0.500000000000000;1595.55950200000,9.08891400000000,0.500000000000000;1723.55346600000,9.20610000000000,0.500000000000000;1844.62883700000,9.32328700000000,0.500000000000000;1803.11728200000,9.29724500000000,0.500000000000000;1931.11124600000,9.41443100000000,0.500000000000000;2007.21576500000,9.49255500000000,0.500000000000000;2093.69817300000,9.51859700000000,0.500000000000000;2138.66902500000,9.51859700000000,0.500000000000000;2180.18058100000,9.53161700000000,0.500000000000000;2225.15143300000,9.54463800000000,0.500000000000000;2263.20369300000,9.55765900000000,0.500000000000000;2346.22680500000,9.57067900000000,0.500000000000000;2422.33132400000,9.57067900000000,0.500000000000000;2512.27302800000,9.62276200000000,0.500000000000000;2598.75543600000,9.64880300000000,0.500000000000000;2678.31925200000,9.64880300000000,0.500000000000000;2764.80166000000,9.67484500000000,0.500000000000000;2892.79562400000,9.72692700000000,0.500000000000000;3013.87099500000,9.75296900000000,0.500000000000000;3103.81270000000,9.76599000000000,0.500000000000000;3224.88807100000,9.76599000000000,0.500000000000000;3262.94033100000,9.79203100000000,0.500000000000000;3387.47499800000,9.79203100000000,0.500000000000000;3432.44585100000,9.80505200000000,0.500000000000000;3518.92825900000,9.81807200000000,0.500000000000000;3598.49207400000,9.84411400000000,0.500000000000000;3723.02674200000,9.83109300000000,0.500000000000000;3767.99759400000,9.85713400000000,0.500000000000000;3892.53226200000,9.85713400000000,0.500000000000000;3930.58452100000,9.88317600000000,0.500000000000000;4020.52622600000,9.89619600000000,0.500000000000000;4134.68300500000,9.93525800000000,0.500000000000000;4266.13626500000,10.0003620000000,0.500000000000000;4356.07796900000,10.0394240000000,0.500000000000000;4477.15334100000,10.1435890000000,0.500000000000000;4646.65886100000,10.2607750000000,0.500000000000000;4812.70508400000,10.3649410000000,0.500000000000000;4940.69904800000,10.4560850000000,0.500000000000000;5009.88497500000,10.5081680000000,0.500000000000000;5096.36738300000,10.4040030000000,0.500000000000000;5224.36134700000,10.2868170000000,0.500000000000000;5386.94827400000,10.1175480000000,0.500000000000000;5521.86083100000,9.98734100000000,0.500000000000000;5691.36635100000,9.80505200000000,0.500000000000000;5815.90101900000,9.67484500000000,0.500000000000000;5943.89498300000,9.55765900000000,0.500000000000000;6002.70302000000,9.46651400000000,0.500000000000000;6000,0.430165000000000,0.500000000000000;5892.00553800000,0.430165000000000,0.500000000000000;5947.35427900000,0.430165000000000,0.500000000000000;5850.49398200000,0.443185000000000,0.500000000000000;5805.52313000000,0.443185000000000,0.500000000000000;5722.50001800000,0.443185000000000,0.500000000000000;5639.47690600000,0.443185000000000,0.500000000000000;5597.96535000000,0.482247000000000,0.500000000000000;5480.34927500000,0.482247000000000,0.500000000000000;5390.40757100000,0.482247000000000,0.500000000000000;5348.89601500000,0.495268000000000,0.500000000000000;5258.95431000000,0.495268000000000,0.500000000000000;5137.87893900000,0.508289000000000,0.500000000000000;5096.36738300000,0.521309000000000,0.500000000000000;5016.80356700000,0.521309000000000,0.500000000000000;4971.83271500000,0.521309000000000,0.500000000000000;4874.97241800000,0.521309000000000,0.500000000000000;4695.08900900000,0.521309000000000,0.500000000000000;4584.39152700000,0.521309000000000,0.500000000000000;4473.69404400000,0.521309000000000,0.500000000000000;4269.59556100000,0.521309000000000,0.500000000000000;4148.52019000000,0.521309000000000,0.500000000000000;4044.74130000000,0.521309000000000,0.500000000000000;3968.63678100000,0.508289000000000,0.500000000000000;3844.10211300000,0.469227000000000,0.500000000000000;3716.10814900000,0.443185000000000,0.500000000000000;3591.57348200000,0.404123000000000,0.500000000000000;3508.55037000000,0.391103000000000,0.500000000000000;3467.03881400000,0.365061000000000,0.500000000000000;3342.50414600000,0.352041000000000,0.500000000000000;3256.02173800000,0.325999000000000,0.500000000000000;3176.45792300000,0.299958000000000,0.500000000000000;3089.97551400000,0.273916000000000,0.500000000000000;3048.46395900000,0.260896000000000,0.500000000000000;3006.95240300000,0.260896000000000,0.500000000000000;2965.44084700000,0.260896000000000,0.500000000000000;2882.41773500000,0.273916000000000,0.500000000000000;2799.39462300000,0.299958000000000,0.500000000000000;2712.91221500000,0.325999000000000,0.500000000000000;2605.67402900000,0.339020000000000,0.500000000000000;2498.43584300000,0.365061000000000,0.500000000000000;2380.81976800000,0.417144000000000,0.500000000000000;2252.82580400000,0.443185000000000,0.500000000000000;2135.20972900000,0.482247000000000,0.500000000000000;2048.72732100000,0.508289000000000,0.500000000000000;2000.29717200000,0.508289000000000,0.500000000000000;1924.19265300000,0.521309000000000,0.500000000000000;1723.55346600000,0.521309000000000,0.500000000000000;1529.83287200000,0.521309000000000,0.500000000000000;1249.62987000000,0.521309000000000,0.500000000000000;983.264053000000,0.521309000000000,0.500000000000000;727.276124000000,0.521309000000000,0.500000000000000;502.421863000000,0.521309000000000,0.500000000000000;503.046458000000,7.84025600000000,0;503.046458000000,9.06709300000000,0;503.046458000000,10.6645370000000,0;503.046458000000,9.75718800000000,0;506.473724000000,8.41533500000000,0;718.964204000000,10.6645370000000,0;931.454684000000,10.6517570000000,0;1167.93602400000,10.6389780000000,0;1448.97182000000,10.6261980000000,0;1311.88118800000,10.6645370000000,0;797.791318000000,10.6645370000000,0;1688.88042700000,10.6773160000000,0;1921.93450100000,10.6773160000000,0;2192.68850000000,10.6773160000000,0;2552.55140900000,10.6900960000000,0;2861.00533100000,10.7028750000000,0;3090.63214000000,10.6773160000000,0;3337.39527800000,10.7412140000000,0;3601.29474500000,10.6900960000000,0;3830.92155400000,10.6773160000000,0;4108.53008400000,10.6773160000000,0;4468.39299300000,10.6773160000000,0;4776.84691500000,10.6773160000000,0;5027.03731900000,10.6773160000000,0;5325.20944400000,10.6773160000000,0;5606.24524000000,10.6645370000000,0;5825.59025100000,10.6645370000000,0;6003.80807300000,10.6773160000000,0;6000.38080700000,9.84664500000000,0;5996.95354200000,10.2172520000000,0;503.046458000000,0.00639000000000000,0;712.109673000000,0.00639000000000000,0;948.591013000000,0.00639000000000000,0;1226.19954300000,-0.00639000000000000,0;1462.68088300000,-0.00639000000000000,0;1664.88956600000,-0.00639000000000000,0;1846.53465300000,-0.00639000000000000,0;2117.28865200000,-0.00639000000000000,0;2302.36100500000,-0.00639000000000000,0;2473.72429600000,-0.00639000000000000,0;2655.36938300000,-0.00639000000000000,0;2816.45087600000,-0.00639000000000000,0;3059.78674800000,0.00639000000000000,0;2943.25971100000,0.00639000000000000,0;3207.15917700000,-0.00639000000000000,0;3429.93145500000,-0.00639000000000000,0;3738.38537700000,-0.00639000000000000,0;4005.71211000000,-0.00639000000000000,0;4290.17517100000,-0.00639000000000000,0;4598.62909400000,0.00639000000000000,0;4797.41051000000,0.00639000000000000,0;4992.76466100000,-0.0319490000000000,0;5277.22772300000,0.00639000000000000,0;5630.23610100000,0.00639000000000000,0;5897.56283300000,0.00639000000000000,0;6003.80807300000,-0.00639000000000000,0;512.799752000000,7.91054300000000,0;588.904271000000,8.05210100000000,0;665.008791000000,8.20652800000000,0;754.950495000000,8.41243100000000,0;824.136422000000,8.55398900000000,0;917.537422000000,8.73415400000000,0;1028.23490500000,8.86284300000000,0;1159.68816500000,8.97866400000000,0;1298.06001800000,9.13309100000000,0;1450.26905600000,9.24891100000000,0;1599.01879800000,9.42907600000000,0;1713.17557700000,9.53202700000000,0;1820.41376300000,9.60924100000000,0;1893.05898600000,9.68645400000000,0;1996.83787600000,9.75079900000000,0;2145.58761800000,9.80227400000000,0;2301.25595200000,9.86661900000000,0;2401.57554600000,9.91809500000000,0;2536.48810200000,9.95670100000000,0;2678.31925200000,9.98243900000000,0;2806.31321600000,10.0596530000000,0;2944.68506900000,10.0853910000000,0;3020.78958800000,10.1626040000000,0;3190.29510800000,10.1239970000000,0;3404.77148000000,10.1368660000000,0;3591.57348200000,10.1754730000000,0;3781.83477900000,10.2012110000000,0;3958.25889200000,10.2269490000000,0;4134.68300500000,10.3556380000000,0;4293.81063600000,10.5100650000000,0;4456.39756300000,10.5229340000000,0;4643.19956400000,10.5486720000000,0;5317.76234800000,10.5100650000000,0;5473.43068200000,10.3813760000000,0;5622.18042400000,10.2012110000000,0;5760.55227700000,10.0467840000000,0;5898.92413000000,9.90522600000000,0;6002.70302000000,9.82801200000000,0;4290.35133900000,10.4071140000000,0];
    
    map_data(:,3) = map_data(:,3) * abs(mod_GDI);

    engine_out = engine_tech_mod( engine_out, mod_GDI > 0, mod_GDI < 0, map_data );
end

%% add or remove continuous cam phasing/discrete variable valve lift effectiveness
if mod_CCP_DVVL	~= 0 
% see REVS_modify_engine() for original implementation
	map_data = [996.635262000000,0.393185000000000,9.50000000000000;1100.94212700000,0,9.50000000000000;1063.93001300000,0.196592000000000,9.50000000000000;996.635262000000,0.655308000000000,9;1225.43741600000,0.0131060000000000,9;1161.50740200000,0.170380000000000,9;1060.56527600000,0.471822000000000,9;1000,0.930537000000000,8.50000000000000;1323.01480500000,0.0131060000000000,8.50000000000000;1114.40107700000,0.615990000000000,8.50000000000000;1215.34320300000,0.314548000000000,8.50000000000000;1000,2.05766700000000,8;1531.62853300000,1.00917400000000,8;1158.14266500000,1.75622500000000,8;1316.28533000000,1.41546500000000,8;1484.52220700000,0.799476000000000,8;1467.69852000000,0.471822000000000,8;1460.96904400000,0.249017000000000,8;1450.87483200000,0,8;1094.21265100000,1.88728700000000,8;1228.80215300000,1.61205800000000,8;1393.67429300000,1.25819100000000,8;1477.79273200000,1.08781100000000,8;800,2.40000000000000,8;1000,2.63433800000000,7.50000000000000;1969.04441500000,1.00917400000000,7.50000000000000;1158.14266500000,2.34600300000000,7.50000000000000;1326.37954200000,2.07077300000000,7.50000000000000;1477.79273200000,1.74311900000000,7.50000000000000;1652.75908500000,1.48099600000000,7.50000000000000;1810.90175000000,1.25819100000000,7.50000000000000;1888.29071300000,0.655308000000000,7.50000000000000;1820.99596200000,0.327654000000000,7.50000000000000;1757.06594900000,0.0262120000000000,7.50000000000000;1000,3.22411500000000,7;1158.14266500000,3.04062900000000,7;1242.26110400000,2.93578000000000,7;2409.82503400000,1.00917400000000,7;1410.49798100000,2.63433800000000,7;1578.73485900000,2.34600300000000,7;1750.33647400000,2.04456100000000,7;1918.57335100000,1.74311900000000,7;2086.81022900000,1.48099600000000,7;2244.95289400000,1.25819100000000,7;2325.70659500000,0.642202000000000,7;2258.41184400000,0.327654000000000,7;2191.11709300000,0.0131060000000000,7;996.635262000000,3.82699900000000,6.50000000000000;1481.15747000000,3.82699900000000,6.50000000000000;1242.26110400000,3.84010500000000,6.50000000000000;1693.13593500000,3.01441700000000,6.50000000000000;1124.49528900000,3.81389300000000,6.50000000000000;1329.74428000000,3.84010500000000,6.50000000000000;1578.73485900000,3.47313200000000,6.50000000000000;2672.27456300000,0.996068000000000,6.50000000000000;1864.73755000000,2.64744400000000,6.50000000000000;2029.60969000000,2.30668400000000,6.50000000000000;2177.65814300000,1.97903000000000,6.50000000000000;2345.89502000000,1.71690700000000,6.50000000000000;2473.75504700000,1.52031500000000,6.50000000000000;2571.33243600000,1.25819100000000,6.50000000000000;2638.62718700000,0.747051000000000,6.50000000000000;2604.97981200000,0.419397000000000,6.50000000000000;2554.50874800000,0,6.50000000000000;800,3.70000000000000,6.50000000000000;996.635262000000,4.22018300000000,6;1070.65948900000,4.23329000000000,6;1238.89636600000,4.31192700000000,6;1481.15747000000,4.37745700000000,6;1336.47375500000,4.31192700000000,6;1985.86810200000,4.16775900000000,6;1662.85329700000,4.29882000000000,6;1827.72543700000,4.23329000000000,6;2069.98654100000,3.97116600000000,6;2086.81022900000,3.85321100000000,6;2086.81022900000,3.64351200000000,6;2117.09286700000,3.32896500000000,6;2117.09286700000,3.14547800000000,6;2144.01076700000,2.98820400000000,6;2497.30821000000,2.14941000000000,6;2288.69448200000,2.64744400000000,6;2399.73082100000,2.38532100000000,6;2867.42934100000,1.00917400000000,6;2618.43876200000,1.82175600000000,6;2722.74562600000,1.49410200000000,6;2810.22880200000,1.21887300000000,6;2776.58142700000,0.0131060000000000,6;2837.14670300000,0.642202000000000,6;2803.49932700000,0.314548000000000,6;1000,4.52162500000000,5.50000000000000;1158.14266500000,4.57405000000000,5.50000000000000;1323.01480500000,4.67889900000000,5.50000000000000;1487.88694500000,4.73132400000000,5.50000000000000;1985.86810200000,4.65268700000000,5.50000000000000;1662.85329700000,4.69200500000000,5.50000000000000;1827.72543700000,4.66579300000000,5.50000000000000;2308.88290700000,3.97116600000000,5.50000000000000;2083.44549100000,4.42988200000000,5.50000000000000;2167.56393000000,4.27260800000000,5.50000000000000;2423.28398400000,3.43381400000000,5.50000000000000;2493.94347200000,3.15858500000000,5.50000000000000;2652.08613700000,2.76540000000000,5.50000000000000;2759.75773900000,2.43774600000000,5.50000000000000;2877.52355300000,2.11009200000000,5.50000000000000;2965.00672900000,1.78243800000000,5.50000000000000;3008.74831800000,1.49410200000000,5.50000000000000;3059.21938100000,1.03538700000000,5.50000000000000;3035.66621800000,1.27129800000000,5.50000000000000;3025.57200500000,0.615990000000000,5.50000000000000;3025.57200500000,0.458716000000000,5.50000000000000;3008.74831800000,0.288336000000000,5.50000000000000;2978.46568000000,0.0131060000000000,5.50000000000000;800,4.50000000000000,5.50000000000000;1000,4.80996100000000,5;1393.67429300000,5.03276500000000,5;1158.14266500000,4.88859800000000,5;1578.73485900000,5.05897800000000,5;1743.60699900000,5.04587200000000,5;1746.97173600000,5.05897800000000,5;1958.95020200000,5.08519000000000,5;3018.84253000000,3.05373500000000,5;2167.56393000000,4.73132400000000,5;2416.55450900000,4.24639600000000,5;2581.42664900000,3.94495400000000,5;2749.66352600000,3.61730000000000,5;2917.90040400000,3.27654000000000,5;3055.85464300000,2.62123200000000,5;3089.50201900000,2.28047200000000,5;3092.86675600000,2.11009200000000,5;3207.26783300000,1.04849300000000,5;3123.14939400000,1.76933200000000,5;3160.16150700000,1.44167800000000,5;3207.26783300000,0.943644000000000,5;3207.26783300000,0.563565000000000,5;3190.44414500000,0.445609000000000,5;3193.80888300000,0.0131060000000000,5;3190.44414500000,0.275229000000000,5;1000,5.15072100000000,4.50000000000000;1158.14266500000,5.16382700000000,4.50000000000000;1326.37954200000,5.19003900000000,4.50000000000000;1491.25168200000,5.22935800000000,4.50000000000000;1662.85329700000,5.26867600000000,4.50000000000000;1831.09017500000,5.26867600000000,4.50000000000000;1989.23284000000,5.29488900000000,4.50000000000000;2164.19919200000,5.13761500000000,4.50000000000000;2332.43607000000,4.90170400000000,4.50000000000000;2500.67294800000,4.57405000000000,4.50000000000000;2584.79138600000,4.46920100000000,4.50000000000000;2985.19515500000,3.98427300000000,4.50000000000000;2749.66352600000,4.27260800000000,4.50000000000000;3156.79677000000,3.14547800000000,4.50000000000000;3055.85464300000,3.64351200000000,4.50000000000000;3119.78465700000,3.31585800000000,4.50000000000000;3325.03364700000,1.48099600000000,4.50000000000000;3207.26783300000,2.64744400000000,4.50000000000000;3277.92732200000,1.97903000000000,4.50000000000000;3321.66891000000,1.31061600000000,4.50000000000000;3375.50471100000,0.825688000000000,4.50000000000000;3372.13997300000,0.655308000000000,4.50000000000000;3392.32839800000,0.498034000000000,4.50000000000000;3388.96366100000,0.327654000000000,4.50000000000000;3422.61103600000,0,4.50000000000000;3409.15208600000,0.157274000000000,4.50000000000000;3237.55047100000,2.30668400000000,4.50000000000000;3338.49259800000,1.15334200000000,4.50000000000000;2837.14670300000,4.16775900000000,4.50000000000000;1000,5.43905600000000,4;1158.14266500000,5.42595000000000,4;1326.37954200000,5.39973800000000,4;1494.61642000000,5.37352600000000,4;1827.72543700000,5.49148100000000,4;1911.84387600000,5.49148100000000,4;1662.85329700000,5.42595000000000,4;1992.59757700000,5.53079900000000,4;2164.19919200000,5.36041900000000,4;2332.43607000000,5.20314500000000,4;2581.42664900000,4.96723500000000,4;2497.30821000000,5.05897800000000,4;3002.01884300000,4.46920100000000,4;2749.66352600000,4.75753600000000,4;2917.90040400000,4.57405000000000,4;3109.69044400000,4.14154700000000,4;3193.80888300000,3.80078600000000,4;3257.73889600000,3.47313200000000,4;3325.03364700000,3.13237200000000,4;3358.68102300000,2.81782400000000,4;3395.69313600000,2.49017000000000,4;3392.32839800000,2.30668400000000,4;3631.22476400000,0.0131060000000000,4;3439.43472400000,1.82175600000000,4;3489.90578700000,1.31061600000000,4;3543.74158800000,0.825688000000000,4;3594.21265100000,0.327654000000000,4;1000,5.77981700000000,3.50000000000000;1158.14266500000,5.70118000000000,3.50000000000000;1323.01480500000,5.59633000000000,3.50000000000000;1491.25168200000,5.53079900000000,3.50000000000000;1985.86810200000,5.75360400000000,3.50000000000000;1662.85329700000,5.62254300000000,3.50000000000000;1831.09017500000,5.68807300000000,3.50000000000000;2507.40242300000,5.38663200000000,3.50000000000000;2167.56393000000,5.62254300000000,3.50000000000000;2329.07133200000,5.49148100000000,3.50000000000000;2991.92463000000,5.00655300000000,3.50000000000000;2665.54508700000,5.25557000000000,3.50000000000000;2837.14670300000,5.12450900000000,3.50000000000000;3348.58681000000,3.94495400000000,3.50000000000000;3116.41991900000,4.63958100000000,3.50000000000000;3234.18573400000,4.29882000000000,3.50000000000000;3533.64737600000,2.77850600000000,3.50000000000000;3419.24629900000,3.47313200000000,3.50000000000000;3483.17631200000,3.09305400000000,3.50000000000000;3560.56527600000,2.49017000000000,3.50000000000000;3600.94212700000,2.14941000000000,3.50000000000000;3644.68371500000,1.82175600000000,3.50000000000000;3695.15477800000,1.49410200000000,3.50000000000000;3745.62584100000,1.17955400000000,3.50000000000000;3786.00269200000,0.825688000000000,3.50000000000000;3823.01480500000,0.314548000000000,3.50000000000000;3856.66218000000,0,3.50000000000000;3809.55585500000,0.563565000000000,3.50000000000000;1000,6.04194000000000,3;1077.38896400000,6.02883400000000,3;1232.16689100000,5.88466600000000,3;1481.15747000000,5.70118000000000,3;1380.21534300000,5.76671000000000,3;1982.50336500000,6.02883400000000,3;1666.21803500000,5.80602900000000,3;1831.09017500000,5.92398400000000,3;2504.03768500000,5.72739200000000,3;2164.19919200000,5.91087800000000,3;2332.43607000000,5.81913500000000,3;2995.28936700000,5.33420700000000,3;2668.90982500000,5.58322400000000,3;2837.14670300000,5.45216300000000,3;3210.63257100000,4.86238500000000,3;3099.59623100000,5.11140200000000,3;3503.36473800000,3.97116600000000,3;3314.93943500000,4.54783700000000,3;3415.88156100000,4.25950200000000,3;3688.42530300000,2.81782400000000,3;3557.20053800000,3.63040600000000,3;3637.95424000000,3.14547800000000,3;3997.98115700000,1.03538700000000,3;3738.89636600000,2.47706400000000,3;3779.27321700000,2.14941000000000,3;3802.82638000000,1.97903000000000,3;3870.12113100000,1.65137600000000,3;3934.05114400000,1.31061600000000,3;4021.53432000000,0.642202000000000,3;4048.45222100000,0.314548000000000,3;4068.64064600000,0,3;800,6.20000000000000,3;1000,6.19921400000000,2.50000000000000;1161.50740200000,6.08125800000000,2.50000000000000;1282.63795400000,6.02883400000000,2.50000000000000;1474.42799500000,5.85845300000000,2.50000000000000;1363.39165500000,5.93709000000000,2.50000000000000;1979.13862700000,6.26474400000000,2.50000000000000;1662.85329700000,5.98951500000000,2.50000000000000;1831.09017500000,6.10747100000000,2.50000000000000;2493.94347200000,6.09436400000000,2.50000000000000;2167.56393000000,6.18610700000000,2.50000000000000;2339.16554500000,6.13368300000000,2.50000000000000;2991.92463000000,5.66186100000000,2.50000000000000;2668.90982500000,5.92398400000000,2.50000000000000;2837.14670300000,5.77981700000000,2.50000000000000;3338.49259800000,4.96723500000000,2.50000000000000;3089.50201900000,5.46526900000000,2.50000000000000;3257.73889600000,5.12450900000000,2.50000000000000;3483.17631200000,4.56094400000000,2.50000000000000;3577.38896400000,4.16775900000000,2.50000000000000;3654.77792700000,3.80078600000000,2.50000000000000;3732.16689100000,3.47313200000000,2.50000000000000;3796.09690400000,3.15858500000000,2.50000000000000;3863.39165500000,2.81782400000000,2.50000000000000;3917.22745600000,2.50327700000000,2.50000000000000;3971.06325700000,2.14941000000000,2.50000000000000;4018.16958300000,1.80865000000000,2.50000000000000;4068.64064600000,1.34993400000000,2.50000000000000;4146.02961000000,0.825688000000000,2.50000000000000;4199.86541000000,0.498034000000000,2.50000000000000;4283.98384900000,0.0131060000000000,2.50000000000000;4240.24226100000,0.235911000000000,2.50000000000000;1000,6.38270000000000,2;1454.23956900000,6.05504600000000,2;1161.50740200000,6.25163800000000,2;1329.74428000000,6.12057700000000,2;1497.98115700000,6.05504600000000,2;1982.50336500000,6.52686800000000,2;1662.85329700000,6.21232000000000,2;1837.81965000000,6.38270000000000,2;2487.21399700000,6.57929200000000,2;2167.56393000000,6.55308000000000,2;2335.80080800000,6.55308000000000,2;2995.28936700000,6.02883400000000,2;2668.90982500000,6.39580600000000,2;2837.14670300000,6.18610700000000,2;3513.45895000000,4.94102200000000,2;3096.23149400000,5.79292300000000,2;3247.64468400000,5.49148100000000,2;3409.15208600000,5.16382700000000,2;3580.75370100000,4.66579300000000,2;3698.51951500000,4.18086500000000,2;3826.37954200000,3.68283100000000,2;3987.88694500000,3.09305400000000,2;3917.22745600000,3.35517700000000,2;4011.44010800000,2.89646100000000,2;4048.45222100000,2.51638300000000,2;4098.92328400000,2.00524200000000,2;4169.58277300000,1.52031500000000,2;4220.05383600000,1.17955400000000,2;4250.33647400000,1.02228000000000,2;4499.32705200000,0.0131060000000000,2;4331.09017500000,0.681520000000000,2;4415.20861400000,0.353866000000000,2;1000,6.51376100000000,1.50000000000000;1242.26110400000,6.39580600000000,1.50000000000000;1161.50740200000,6.42201800000000,1.50000000000000;1454.23956900000,6.27785100000000,1.50000000000000;1326.37954200000,6.33027500000000,1.50000000000000;1497.98115700000,6.29095700000000,1.50000000000000;1410.49798100000,6.29095700000000,1.50000000000000;1982.50336500000,6.76277900000000,1.50000000000000;1662.85329700000,6.44823100000000,1.50000000000000;1831.09017500000,6.61861100000000,1.50000000000000;2497.30821000000,7.06422000000000,1.50000000000000;2167.56393000000,6.84141500000000,1.50000000000000;2332.43607000000,6.95937100000000,1.50000000000000;2672.27456300000,6.88073400000000,1.50000000000000;3002.01884300000,6.48754900000000,1.50000000000000;2840.51144000000,6.68414200000000,1.50000000000000;3493.27052500000,5.49148100000000,1.50000000000000;3173.62045800000,6.15989500000000,1.50000000000000;3341.85733500000,5.81913500000000,1.50000000000000;3614.40107700000,5.00655300000000,1.50000000000000;4109.01749700000,3.05373500000000,1.50000000000000;3742.26110400000,4.50851900000000,1.50000000000000;3870.12113100000,3.99737900000000,1.50000000000000;3997.98115700000,3.49934500000000,1.50000000000000;4146.02961000000,2.66055000000000,1.50000000000000;4149.39434700000,2.51638300000000,1.50000000000000;4196.50067300000,2.00524200000000,1.50000000000000;4233.51278600000,1.84796900000000,1.50000000000000;4246.97173600000,1.67758800000000,1.50000000000000;4314.26648700000,1.33682800000000,1.50000000000000;4327.72543700000,1.19266100000000,1.50000000000000;4415.20861400000,0.851900000000000,1.50000000000000;4499.32705200000,0.524246000000000,1.50000000000000;4566.62180300000,0.196592000000000,1.50000000000000;4617.09286700000,0,1.50000000000000;1003.36473800000,6.68414200000000,1;1161.50740200000,6.63171700000000,1;1242.26110400000,6.57929200000000,1;1329.74428000000,6.57929200000000,1;1410.49798100000,6.55308000000000,1;1494.61642000000,6.53997400000000,1;2053.16285300000,7.07732600000000,1;1666.21803500000,6.72346000000000,1;1841.18438800000,6.88073400000000,1;2493.94347200000,7.36566200000000,1;2251.68236900000,7.18217600000000,1;2998.65410500000,7.01179600000000,1;2668.90982500000,7.20838800000000,1;2837.14670300000,7.11664500000000,1;3503.36473800000,6.00262100000000,1;3176.98519500000,6.65792900000000,1;3345.22207300000,6.33027500000000,1;4250.33647400000,3.01441700000000,1;3580.75370100000,5.66186100000000,1;3708.61372800000,5.16382700000000,1;3829.74428000000,4.66579300000000,1;3950.87483200000,4.16775900000000,1;4085.46433400000,3.68283100000000,1;4169.58277300000,3.34207100000000,1;4250.33647400000,2.83093100000000,1;4263.79542400000,2.67365700000000,1;4263.79542400000,2.50327700000000,1;4280.61911200000,2.33289600000000,1;4280.61911200000,2.17562300000000,1;4297.44279900000,2.00524200000000,1;4331.09017500000,1.84796900000000,1;4398.38492600000,1.52031500000000,1;4499.32705200000,1.00917400000000,1;4583.44549100000,0.681520000000000,1;4667.56393000000,0.353866000000000,1;4755.04710600000,-0.0131060000000000,1;996.635262000000,6.82830900000000,0.500000000000000;1164.87214000000,6.80209700000000,0.500000000000000;1326.37954200000,6.77588500000000,0.500000000000000;1410.49798100000,6.73656600000000,0.500000000000000;1491.25168200000,6.77588500000000,0.500000000000000;1985.86810200000,7.48361700000000,0.500000000000000;1662.85329700000,6.94626500000000,0.500000000000000;1827.72543700000,7.19528200000000,0.500000000000000;2497.30821000000,7.68021000000000,0.500000000000000;2164.19919200000,7.53604200000000,0.500000000000000;2332.43607000000,7.60157300000000,0.500000000000000;2998.65410500000,7.47051100000000,0.500000000000000;2753.02826400000,7.57536000000000,0.500000000000000;2628.53297400000,7.60157300000000,0.500000000000000;2870.79407800000,7.52293600000000,0.500000000000000;3748.99057900000,6.00262100000000,0.500000000000000;3176.98519500000,7.14285700000000,0.500000000000000;3341.85733500000,6.81520300000000,0.500000000000000;3510.09421300000,6.47444300000000,0.500000000000000;3799.46164200000,5.60943600000000,0.500000000000000;3846.56796800000,5.15072100000000,0.500000000000000;3913.86271900000,4.82306700000000,0.500000000000000;3997.98115700000,4.49541300000000,0.500000000000000;4162.85329700000,4.16775900000000,0.500000000000000;4257.06594900000,3.95806000000000,0.500000000000000;4310.90175000000,3.49934500000000,0.500000000000000;4368.10228800000,3.04062900000000,0.500000000000000;4368.10228800000,2.84403700000000,0.500000000000000;4388.29071300000,2.67365700000000,0.500000000000000;4388.29071300000,2.34600300000000,0.500000000000000;4408.47913900000,1.99213600000000,0.500000000000000;4472.40915200000,1.66448200000000,0.500000000000000;4502.69179000000,1.50720800000000,0.500000000000000;4751.68236900000,1.00917400000000,0.500000000000000;4808.88290700000,0.524246000000000,0.500000000000000;4869.44818300000,-0.0131060000000000,0.500000000000000;4586.81022900000,1.33682800000000,0.500000000000000;4670.92866800000,1.17955400000000,0.500000000000000;800,6.80000000000000,0.500000000000000;993.270525000000,7.99475800000000,0;993.270525000000,10.0131060000000,0;993.270525000000,10.8912190000000,0;1709.95962300000,10.8912190000000,0;2544.41453600000,10.8912190000000,0;3452.89367400000,10.9043250000000,0;4384.92597600000,10.9043250000000,0;5313.59354000000,10.9043250000000,0;5754.37415900000,10.8912190000000,0;6010.09421300000,10.9043250000000,0;6016.82368800000,9.39711700000000,0;6013.45895000000,7.81127100000000,0;6016.82368800000,6.12057700000000,0;6013.45895000000,4.19397100000000,0;6013.45895000000,2.51638300000000,0;6013.45895000000,1.14023600000000,0;6013.45895000000,-0.0131060000000000,0;5528.93674300000,-0.0262120000000000,0;5104.97981200000,-0.0131060000000000,0;993.270525000000,7.02490200000000,0;1111.03633900000,6.98558300000000,0;1232.16689100000,6.93315900000000,0;1346.56796800000,6.93315900000000,0;1440.78061900000,6.95937100000000,0;1548.45222100000,7.05111400000000,0;1669.58277300000,7.20838800000000,0;1790.71332400000,7.35255600000000,0;1891.65545100000,7.56225400000000,0;1989.23284000000,7.69331600000000,0;2127.18707900000,7.71952800000000,0;2255.04710600000,7.78505900000000,0;2379.54239600000,7.82437700000000,0;2487.21399700000,7.90301400000000,0;2615.07402400000,7.85059000000000,0;2769.85195200000,7.77195300000000,0;2904.44145400000,7.75884700000000,0;3005.38358000000,7.77195300000000,0;3143.33782000000,7.44429900000000,0;3281.29205900000,7.23460000000000,0;3372.13997300000,7.01179600000000,0;3500,6.78899100000000,0;3614.40107700000,6.53997400000000,0;3745.62584100000,6.30406300000000,0;3823.01480500000,6.15989500000000,0;3860.02691800000,5.62254300000000,0;3930.68640600000,5.12450900000000,0;4004.71063300000,4.80996100000000,0;4105.65275900000,4.53473100000000,0;4223.41857300000,4.33813900000000,0;4314.26648700000,4.14154700000000,0;4388.29071300000,3.46002600000000,0;4358.00807500000,3.78768000000000,0;4425.30282600000,2.98820400000000,0;4465.67967700000,2.67365700000000,0;4442.12651400000,2.38532100000000,0;4482.50336500000,1.92660600000000,0;4580.08075400000,1.63827000000000,0;4694.48183000000,1.37614700000000,0;4802.15343200000,1.16644800000000,0;4855.98923300000,0.694626000000000,0;4896.36608300000,0.262123000000000,0;4919.91924600000,0,0;1497.98115700000,9.67234600000000,0;2483.84926000000,9.48886000000000,0;3792.73216700000,9.50196600000000,0;4738.22341900000,9.34469200000000,0;5438.08882900000,9.12188700000000,0;5485.19515500000,6.47444300000000,0;5535.66621800000,3.09305400000000,0;4956.93135900000,2.43774600000000,0;5623.14939400000,1.36304100000000,0;4731.49394300000,4.11533400000000,0;4324.36070000000,6.02883400000000,0;4024.89905800000,7.79816500000000,0;3102.96096900000,8.68938400000000,0;1925.30282600000,8.83355200000000,0;1366.75639300000,8.12581900000000,0;1201.88425300000,9.12188700000000,0;4755.04710600000,7.96854500000000,0;5081.42664900000,5.81913500000000,0;800,7.40000000000000,0;800,9,0;800,11,0];
    
    map_data(:,3) = map_data(:,3) * abs(mod_CCP_DVVL);

    engine_out = engine_tech_mod( engine_out, mod_CCP_DVVL > 0, mod_CCP_DVVL < 0, map_data );
end

%% add or remove discrete cam phasing/continuous variable valve lift effectiveness
if mod_DCP_CVVL ~= 0 
% see REVS_modify_engine() for original implementation
  	map_data = [1000,0.158730000000000,15.5000000000000;1043.94861400000,0,15.5000000000000;1000,0.291005000000000,15;1108.18120400000,0,15;1064.23259000000,0.132275000000000,15;1003.38066300000,0.423280000000000,14.5000000000000;1060.85192700000,0.264550000000000,14.5000000000000;1131.84584200000,0.105820000000000,14.5000000000000;1179.17511800000,0,14.5000000000000;1000,0.542328000000000,14;1226.50439500000,0.0132280000000000,14;1064.23259000000,0.396825000000000,14;1128.46517900000,0.238095000000000,14;1280.59499700000,0.0132280000000000,13.5000000000000;1212.98174400000,0.198413000000000,13.5000000000000;1000,0.661376000000000,13.5000000000000;1145.36849200000,0.370370000000000,13.5000000000000;1060.85192700000,0.542328000000000,13.5000000000000;1358.35023700000,0,13;1212.98174400000,0.370370000000000,13;1128.46517900000,0.529101000000000,13;1000,0.873016000000000,13;1057.47126400000,0.714286000000000,13;1000,1.03174600000000,12.5000000000000;1108.18120400000,0.780423000000000,12.5000000000000;1216.36240700000,0.502646000000000,12.5000000000000;1324.54361100000,0.251323000000000,12.5000000000000;1419.20216400000,0,12.5000000000000;800,1.60000000000000,12.5000000000000;1003.38066300000,1.34920600000000,12;1162.27180500000,0.978836000000000,12;1270.45300900000,0.648148000000000,12;1378.63421200000,0.330688000000000,12;1483.43475300000,0,12;1000,1.67989400000000,11.5000000000000;1169.03313000000,1.32275100000000,11.5000000000000;1327.92427300000,0.978836000000000,11.5000000000000;1436.10547700000,0.648148000000000,11.5000000000000;1584.85463200000,0.317460000000000,11.5000000000000;1726.84246100000,0.0132280000000000,11.5000000000000;1000,2.02381000000000,11;1973.63083200000,0.0132280000000000,11;1169.03313000000,1.66666700000000,11;1348.20824900000,1.28306900000000,11;1581.47396900000,0.820106000000000,11;1750.50709900000,0.462963000000000,11;1496.95740400000,0.978836000000000,11;1862.06896600000,0.224868000000000,11;1000,2.34127000000000,10.5000000000000;2227.18052700000,0.0132280000000000,10.5000000000000;1246.78837100000,1.97089900000000,10.5000000000000;1415.82150100000,1.64021200000000,10.5000000000000;1581.47396900000,1.30952400000000,10.5000000000000;1750.50709900000,0.978836000000000,10.5000000000000;1916.15956700000,0.648148000000000,10.5000000000000;2081.81203500000,0.317460000000000,10.5000000000000;1003.38066300000,2.65873000000000,10;1158.89114300000,2.43386200000000,10;1331.30493600000,2.20899500000000,10;1496.95740400000,1.97089900000000,10;1665.99053400000,1.64021200000000,10;1831.64300200000,1.30952400000000,10;2081.81203500000,0.820106000000000,10;2467.20757300000,0.0132280000000000,10;2321.83908000000,0.317460000000000,10;2227.18052700000,0.542328000000000,10;1005.54491900000,2.99004200000000,9.50000000000000;1162.50876800000,2.84179400000000,9.50000000000000;1411.60357100000,2.57225200000000,9.50000000000000;1667.52289100000,2.14098500000000,9.50000000000000;1923.44221000000,1.64233300000000,9.50000000000000;2165.71249800000,1.14368100000000,9.50000000000000;2418.21956000000,0.658506000000000,9.50000000000000;2622.95501500000,0.0250820000000000,9.50000000000000;2486.46471200000,0.483303000000000,9.50000000000000;2571.77115100000,0.146376000000000,9.50000000000000;1005.54491900000,3.32696900000000,9;1162.50876800000,3.24610700000000,9;1329.70938900000,3.11133600000000,9;1496.91001100000,2.97656500000000,9;1664.11063300000,2.63963800000000,9;1913.20543700000,2.14098500000000,9;2169.12475600000,1.64233300000000,9;2336.32537800000,1.30540600000000,9;2486.46471200000,0.995433000000000,9;2571.77115100000,0.645028000000000,9;2742.38403100000,0.0116050000000000,9;2646.84081800000,0.362010000000000,9;1002.13266100000,3.66389600000000,8.50000000000000;1162.50876800000,3.55608000000000,8.50000000000000;1411.60357100000,3.36740000000000,8.50000000000000;1493.49775400000,3.31349200000000,8.50000000000000;1664.11063300000,3.09785900000000,8.50000000000000;1756.24158800000,2.97656500000000,8.50000000000000;2001.92413400000,2.47791300000000,8.50000000000000;1916.61769400000,2.63963800000000,8.50000000000000;2169.12475600000,1.97926000000000,8.50000000000000;2094.05508900000,2.19489400000000,8.50000000000000;2332.91312000000,1.64233300000000,8.50000000000000;2503.52600000000,1.31888300000000,8.50000000000000;2882.28659200000,0.0116050000000000,8.50000000000000;2626.36727300000,0.981956000000000,8.50000000000000;2708.26145500000,0.645028000000000,8.50000000000000;2756.03306100000,0.496780000000000,8.50000000000000;2837.92724300000,0.146376000000000,8.50000000000000;1002.13266100000,4.00082400000000,8;1159.09651000000,3.89300700000000,8;1329.70938900000,3.77171300000000,8;1486.67323800000,3.66389600000000,8;1664.11063300000,3.43478600000000,8;1834.72351200000,3.20567500000000,8;1913.20543700000,3.11133600000000,8;1991.68736100000,3.00351900000000,8;2336.32537800000,1.96578300000000,8;2104.29186200000,2.63963800000000,8;2216.89636200000,2.30271000000000,8;2500.11374200000,1.62885600000000,8;2564.94663600000,1.48060800000000,8;2691.20016700000,1.14368100000000,8;2783.33112200000,0.820231000000000,8;2988.06657700000,-0.00187200000000000,8;2953.94400100000,0.159853000000000,8;2861.81304600000,0.537212000000000,8;1002.13266100000,4.27036500000000,7.50000000000000;1165.92102500000,4.16254900000000,7.50000000000000;1496.91001100000,4.00082400000000,7.50000000000000;1329.70938900000,4.06820900000000,7.50000000000000;2001.92413400000,3.47521700000000,7.50000000000000;1664.11063300000,3.82562100000000,7.50000000000000;1831.31125500000,3.66389600000000,7.50000000000000;2169.12475600000,3.13829000000000,7.50000000000000;2244.19442300000,2.97656500000000,7.50000000000000;2322.67634700000,2.63963800000000,7.50000000000000;2407.98278700000,2.31618700000000,7.50000000000000;2503.52600000000,1.97926000000000,7.50000000000000;2626.36727300000,1.64233300000000,7.50000000000000;2756.03306100000,1.31888300000000,7.50000000000000;2872.04981900000,0.981956000000000,7.50000000000000;2957.35625900000,0.645028000000000,7.50000000000000;3022.18915300000,0.375487000000000,7.50000000000000;3131.38139600000,-0.00187200000000000,7.50000000000000;3080.19753200000,0.173330000000000,7.50000000000000;1000,4.51058200000000,7;1162.27180500000,4.43121700000000,7;1331.30493600000,4.40476200000000,7;1496.95740400000,4.33862400000000,7;1665.99053400000,4.19312200000000,7;1831.64300200000,4.10052900000000,7;2000.67613300000,3.96825400000000,7;2169.70926300000,3.63756600000000,7;2335.36173100000,3.30687800000000,7;2487.49154800000,2.97619000000000,7;2572.00811400000,2.64550300000000,7;2659.90534100000,2.31481500000000,7;2737.66058100000,1.97089900000000,7;2825.55780900000,1.64021200000000,7;2913.45503700000,1.30952400000000,7;2991.21027700000,0.992063000000000,7;3075.72684200000,0.648148000000000,7;3156.86274500000,0.317460000000000,7;3248.14063600000,0,7;3207.57268400000,0.158730000000000,7;1000,4.77513200000000,6.50000000000000;1077.75524000000,4.73545000000000,6.50000000000000;1165.65246800000,4.73545000000000,6.50000000000000;1331.30493600000,4.69576700000000,6.50000000000000;1496.95740400000,4.65608500000000,6.50000000000000;1665.99053400000,4.52381000000000,6.50000000000000;1750.50709900000,4.51058200000000,6.50000000000000;2000.67613300000,4.29894200000000,6.50000000000000;1831.64300200000,4.43121700000000,6.50000000000000;2169.70926300000,4.11375700000000,6.50000000000000;2737.66058100000,2.97619000000000,6.50000000000000;2419.87829600000,3.63756600000000,6.50000000000000;2585.53076400000,3.30687800000000,6.50000000000000;2791.75118300000,2.80423300000000,6.50000000000000;2818.79648400000,2.64550300000000,6.50000000000000;2991.21027700000,1.97089900000000,6.50000000000000;2906.69371200000,2.31481500000000,6.50000000000000;3038.53955400000,1.64021200000000,6.50000000000000;3089.24949300000,1.30952400000000,6.50000000000000;3106.15280600000,1.15079400000000,6.50000000000000;3123.05611900000,0.978836000000000,6.50000000000000;3170.38539600000,0.820106000000000,6.50000000000000;3383.36714000000,0,6.50000000000000;3251.52129800000,0.489418000000000,6.50000000000000;3342.79918900000,0.145503000000000,6.50000000000000;1000,5.03968300000000,6;1496.95740400000,5,6;1246.78837100000,5,6;1081.13590300000,5,6;1371.87288700000,5,6;1665.99053400000,4.86772500000000,6;1578.09330600000,4.92063500000000,6;1831.64300200000,4.76190500000000,6;2000.67613300000,4.62963000000000,6;2169.70926300000,4.44444400000000,6;2335.36173100000,4.19312200000000,6;2504.39486100000,3.96825400000000,6;2670.04732900000,3.63756600000000,6;2839.08046000000,3.30687800000000,6;2923.59702500000,3.13492100000000,6;2991.21027700000,2.97619000000000,6;2423.25895900000,4.10052900000000,6;3494.92900600000,-0.0132280000000000,6;3021.63624100000,2.80423300000000,6;3038.53955400000,2.64550300000000,6;3058.82352900000,2.47354500000000,6;3106.15280600000,2.14285700000000,6;3139.95943200000,1.81216900000000,6;3190.66937100000,1.48148100000000,6;3224.47599700000,1.15079400000000,6;3241.37931000000,0.978836000000000,6;3322.51521300000,0.648148000000000,6;3407.03177800000,0.317460000000000,6;800,5,6;1000,5.23809500000000,5.50000000000000;1162.27180500000,5.22486800000000,5.50000000000000;1331.30493600000,5.15873000000000,5.50000000000000;1496.95740400000,5.11904800000000,5.50000000000000;1662.60987200000,5.09259300000000,5.50000000000000;1831.64300200000,5.02645500000000,5.50000000000000;1990.53414500000,5,5.50000000000000;2162.94793800000,4.82804200000000,5.50000000000000;2331.98106800000,4.65608500000000,5.50000000000000;2494.25287400000,4.49735400000000,5.50000000000000;3004.73292800000,3.46560800000000,5.50000000000000;2670.04732900000,4.14021200000000,5.50000000000000;2839.08046000000,3.79629600000000,5.50000000000000;3150.10142000000,2.89682500000000,5.50000000000000;3089.24949300000,3.13492100000000,5.50000000000000;3207.57268400000,2.31481500000000,5.50000000000000;3275.18593600000,1.81216900000000,5.50000000000000;3329.27653800000,1.45502600000000,5.50000000000000;3366.46382700000,1.11111100000000,5.50000000000000;3383.36714000000,0.978836000000000,5.50000000000000;3620.01352300000,-0.0132280000000000,5.50000000000000;3467.88370500000,0.648148000000000,5.50000000000000;3545.63894500000,0.317460000000000,5.50000000000000;1000,5.46296300000000,5;1077.75524000000,5.44973500000000,5;1277.21433400000,5.33068800000000,5;1172.41379300000,5.41005300000000,5;1415.82150100000,5.29100500000000,5;1581.47396900000,5.25132300000000,5;1750.50709900000,5.22486800000000,5;1835.02366500000,5.22486800000000,5;2004.05679500000,5.19841300000000,5;2494.25287400000,5,5;2169.70926300000,5.11904800000000,5;2335.36173100000,5.05291000000000,5;2592.29208900000,4.80158700000000,5;2757.94455700000,4.47089900000000,5;3011.49425300000,3.96825400000000,5;2883.02907400000,4.23280400000000,5;3265.04394900000,2.94973500000000,5;3106.15280600000,3.57142900000000,5;3197.43069600000,3.22751300000000,5;3417.17376600000,1.77248700000000,5;3312.37322500000,2.50000000000000,5;3376.60581500000,2.03703700000000,5;3505.07099400000,0.965608000000000,5;3461.12238000000,1.30952400000000,5;3748.47870200000,0.0132280000000000,5;3582.82623400000,0.648148000000000,5;3670.72346200000,0.330688000000000,5;1000,5.75396800000000,4.50000000000000;1466.53144000000,5.39682500000000,4.50000000000000;1165.65246800000,5.63492100000000,4.50000000000000;1331.30493600000,5.48941800000000,4.50000000000000;1496.95740400000,5.38359800000000,4.50000000000000;1588.23529400000,5.39682500000000,4.50000000000000;1750.50709900000,5.38359800000000,4.50000000000000;1916.15956700000,5.39682500000000,4.50000000000000;2494.25287400000,5.26455000000000,4.50000000000000;2169.70926300000,5.34391500000000,4.50000000000000;2335.36173100000,5.27777800000000,4.50000000000000;2747.80256900000,4.97354500000000,4.50000000000000;2639.62136600000,5.10582000000000,4.50000000000000;2997.97160200000,4.48412700000000,4.50000000000000;2896.55172400000,4.69576700000000,4.50000000000000;3379.98647700000,3.02910100000000,4.50000000000000;3129.81744400000,3.99470900000000,4.50000000000000;3254.90196100000,3.50529100000000,4.50000000000000;3620.01352300000,0.978836000000000,4.50000000000000;3447.59973000000,2.38095200000000,4.50000000000000;3532.11629500000,1.67989400000000,4.50000000000000;3870.18255600000,0,4.50000000000000;3697.76876300000,0.648148000000000,4.50000000000000;3778.90466500000,0.317460000000000,4.50000000000000;800,5.90000000000000,4.50000000000000;1000,6.01851900000000,4;1463.15077800000,5.50264600000000,4;1162.27180500000,5.82010600000000,4;1331.30493600000,5.66137600000000,4;1496.95740400000,5.48941800000000,4;1581.47396900000,5.52910100000000,4;1750.50709900000,5.54232800000000,4;1916.15956700000,5.59523800000000,4;1997.29547000000,5.59523800000000,4;2504.39486100000,5.50264600000000,4;2169.70926300000,5.55555600000000,4;2338.74239400000,5.52910100000000,4;2997.97160200000,4.98677200000000,4;3498.30966900000,3.02910100000000,4;2700.47329300000,5.27777800000000,4;2855.98377300000,5.10582000000000,4;3133.19810700000,4.47089900000000,4;3251.52129800000,3.98148100000000,4;3383.36714000000,3.47883600000000,4;3741.71737700000,1.05820100000000,4;3562.54225800000,2.47354500000000,4;3643.67816100000,1.81216900000000,4;3995.26707200000,0.0132280000000000,4;3836.37593000000,0.648148000000000,4;3924.27315800000,0.317460000000000,4;1003.38066300000,6.11111100000000,3.50000000000000;1415.82150100000,5.68783100000000,3.50000000000000;1165.65246800000,5.96560800000000,3.50000000000000;1294.11764700000,5.83333300000000,3.50000000000000;1496.95740400000,5.64814800000000,3.50000000000000;1655.84854600000,5.68783100000000,3.50000000000000;1835.02366500000,5.75396800000000,3.50000000000000;1993.91480700000,5.79365100000000,3.50000000000000;2504.39486100000,5.75396800000000,3.50000000000000;2250.84516600000,5.75396800000000,3.50000000000000;2997.97160200000,5.25132300000000,3.50000000000000;2754.56389500000,5.48941800000000,3.50000000000000;3150.10142000000,4.94709000000000,3.50000000000000;3613.25219700000,3.04232800000000,3.50000000000000;3281.94726200000,4.41798900000000,3.50000000000000;3440.83840400000,3.74338600000000,3.50000000000000;3856.65990500000,1.08465600000000,3.50000000000000;3697.76876300000,2.42063500000000,3.50000000000000;3782.28532800000,1.74603200000000,3.50000000000000;4120.35158900000,0,3.50000000000000;3978.36375900000,0.582011000000000,3.50000000000000;1000,6.25661400000000,3;1493.57674100000,5.75396800000000,3;1165.65246800000,6.08465600000000,3;1334.68559800000,5.92592600000000,3;1669.37119700000,5.85978800000000,3;1835.02366500000,5.96560800000000,3;1983.77281900000,6.03174600000000,3;2494.25287400000,6.01851900000000,3;2169.70926300000,6.01851900000000,3;2335.36173100000,6.03174600000000,3;2994.59094000000,5.48941800000000,3;3265.04394900000,4.96031700000000,3;3133.19810700000,5.22486800000000,3;2673.42799200000,5.82010600000000,3;2839.08046000000,5.66137600000000,3;3745.09803900000,3.05555600000000,3;3417.17376600000,4.32539700000000,3;3592.96822200000,3.63756600000000,3;4002.02839800000,0.992063000000000,3;3829.61460400000,2.31481500000000,3;3893.84719400000,1.81216900000000,3;3961.46044600000,1.30952400000000,3;4255.57809300000,0,3;4086.54496300000,0.661376000000000,3;4198.10682900000,0.211640000000000,3;1000,6.38888900000000,2.50000000000000;1496.95740400000,5.89947100000000,2.50000000000000;1165.65246800000,6.21693100000000,2.50000000000000;1334.68559800000,6.04497400000000,2.50000000000000;1811.35902600000,6.09788400000000,2.50000000000000;1665.99053400000,5.99206300000000,2.50000000000000;1977.01149400000,6.26984100000000,2.50000000000000;2487.49154800000,6.52116400000000,2.50000000000000;2173.08992600000,6.34920600000000,2.50000000000000;2338.74239400000,6.44179900000000,2.50000000000000;2761.32522000000,6.00529100000000,2.50000000000000;2639.62136600000,6.24338600000000,2.50000000000000;2997.97160200000,5.75396800000000,2.50000000000000;2883.02907400000,5.87301600000000,2.50000000000000;3379.98647700000,5,2.50000000000000;3183.90804600000,5.38359800000000,2.50000000000000;3863.42123100000,3.04232800000000,2.50000000000000;3518.59364400000,4.39153400000000,2.50000000000000;3684.24611200000,3.73015900000000,2.50000000000000;4120.35158900000,0.992063000000000,2.50000000000000;3954.69912100000,2.35449700000000,2.50000000000000;4029.07369800000,1.74603200000000,2.50000000000000;4062.88032500000,1.41534400000000,2.50000000000000;4370.52062200000,0,2.50000000000000;4221.77146700000,0.595238000000000,2.50000000000000;4306.28803200000,0.264550000000000,2.50000000000000;1000,6.46825400000000,2;1081.13590300000,6.41534400000000,2;1250.16903300000,6.24338600000000,2;1500.33806600000,6.04497400000000,2;1415.82150100000,6.08465600000000,2;1665.99053400000,6.21693100000000,2;1838.40432700000,6.38888900000000,2;2000.67613300000,6.54761900000000,2;2494.25287400000,6.99735400000000,2;2173.08992600000,6.71957700000000,2;2335.36173100000,6.87830700000000,2;2673.42799200000,6.64021200000000,2;2839.08046000000,6.32275100000000,2;3494.92900600000,4.98677200000000,2;3092.63015600000,5.82010600000000,2;3325.89587600000,5.31746000000000,2;3995.26707200000,3.00264600000000,2;3660.58147400000,4.32539700000000,2;3829.61460400000,3.66402100000000,2;4012.17038500000,2.83068800000000,2;4116.97092600000,1.83862400000000,2;4062.88032500000,2.34127000000000,2;4248.81676800000,1.00529100000000,2;4198.10682900000,1.34920600000000,2;4502.36646400000,0,2;4363.75929700000,0.515873000000000,2;4414.46923600000,0.357143000000000,2;1000,6.61375700000000,1.50000000000000;1165.65246800000,6.48148100000000,1.50000000000000;1500.33806600000,6.28306900000000,1.50000000000000;1334.68559800000,6.34920600000000,1.50000000000000;2501.01419900000,7.24867700000000,1.50000000000000;1669.37119700000,6.44179900000000,1.50000000000000;1919.54023000000,6.67989400000000,1.50000000000000;2173.08992600000,6.94444400000000,1.50000000000000;2342.12305600000,7.11640200000000,1.50000000000000;2673.42799200000,7.08994700000000,1.50000000000000;3494.92900600000,5.48941800000000,1.50000000000000;2926.97768800000,6.65343900000000,1.50000000000000;3173.76605800000,6.15079400000000,1.50000000000000;4113.59026400000,3.00264600000000,1.50000000000000;4079.78363800000,3.16137600000000,1.50000000000000;3647.05882400000,4.84127000000000,1.50000000000000;3789.04665300000,4.31216900000000,1.50000000000000;3893.84719400000,3.90211600000000,1.50000000000000;3995.26707200000,3.49206300000000,1.50000000000000;3342.79918900000,5.82010600000000,1.50000000000000;4147.39689000000,2.67195800000000,1.50000000000000;4147.39689000000,2.50000000000000,1.50000000000000;4184.58417800000,2.16931200000000,1.50000000000000;4194.72616600000,1.99735400000000,1.50000000000000;4252.19743100000,1.65343900000000,1.50000000000000;4316.43002000000,1.33597900000000,1.50000000000000;4329.95267100000,1.16402100000000,1.50000000000000;4417.84989900000,0.846561000000000,1.50000000000000;4617.30899300000,-0.0132280000000000,1.50000000000000;4532.79242700000,0.343915000000000,1.50000000000000;1003.38066300000,6.74603200000000,1;1081.13590300000,6.71957700000000,1;1250.16903300000,6.61375700000000,1;1469.91210300000,6.53439200000000,1;1331.30493600000,6.58730200000000,1;2487.49154800000,7.47354500000000,1;1669.37119700000,6.71957700000000,1;1916.15956700000,6.94444400000000,1;2173.08992600000,7.18254000000000,1;2335.36173100000,7.35449700000000,1;3001.35226500000,7.01058200000000,1;2673.42799200000,7.31481500000000,1;2842.46112200000,7.14285700000000,1;3511.83231900000,5.96560800000000,1;3180.52738300000,6.65343900000000,1;3349.56051400000,6.32275100000000,1;4258.95875600000,3.01587300000000,1;3670.72346200000,5.34391500000000,1;3832.99526700000,4.65608500000000,1;4002.02839800000,3.99470900000000,1;4177.82285300000,3.33333300000000,1;4279.24273200000,2.51322800000000,1;4286.00405700000,2.18254000000000,1;4306.28803200000,2.01058200000000,1;4566.59905300000,0.701058000000000,1;4397.56592300000,1.52116400000000,1;4495.60513900000,1.01851900000000,1;4749.15483400000,0,1;4671.39959400000,0.357143000000000,1;1003.38066300000,6.85185200000000,0.500000000000000;1165.65246800000,6.83862400000000,0.500000000000000;1496.95740400000,6.77248700000000,0.500000000000000;1334.68559800000,6.77248700000000,0.500000000000000;1987.15348200000,7.48677200000000,0.500000000000000;1669.37119700000,6.94444400000000,0.500000000000000;1831.64300200000,7.20899500000000,0.500000000000000;2497.63353600000,7.75132300000000,0.500000000000000;2173.08992600000,7.57936500000000,0.500000000000000;2335.36173100000,7.67195800000000,0.500000000000000;3001.35226500000,7.46031700000000,0.500000000000000;2673.42799200000,7.64550300000000,0.500000000000000;2839.08046000000,7.53968300000000,0.500000000000000;3755.24002700000,5.99206300000000,0.500000000000000;3261.66328600000,6.98412700000000,0.500000000000000;3508.45165700000,6.48148100000000,0.500000000000000;3812.71129100000,5.48941800000000,0.500000000000000;3866.80189300000,4.98677200000000,0.500000000000000;3947.93779600000,4.65608500000000,0.500000000000000;3995.26707200000,4.49735400000000,0.500000000000000;4245.43610500000,4.00793700000000,0.500000000000000;4083.16430000000,4.32539700000000,0.500000000000000;4164.30020300000,4.16666700000000,0.500000000000000;4864.09736300000,0,0.500000000000000;4286.00405700000,3.66402100000000,0.500000000000000;4333.33333300000,3.33333300000000,0.500000000000000;4370.52062200000,3.04232800000000,0.500000000000000;4367.13995900000,2.84391500000000,0.500000000000000;4387.42393500000,2.67195800000000,0.500000000000000;4380.66261000000,2.34127000000000,0.500000000000000;4400.94658600000,2.16931200000000,0.500000000000000;4400.94658600000,1.99735400000000,0.500000000000000;4502.36646400000,1.52116400000000,0.500000000000000;4749.15483400000,1.01851900000000,0.500000000000000;4620.68965500000,1.26984100000000,0.500000000000000;4789.72278600000,0.687831000000000,0.500000000000000;4833.67140000000,0.357143000000000,0.500000000000000;2588.91142700000,7.68518500000000,0.500000000000000;2754.56389500000,7.60582000000000,0.500000000000000;2926.97768800000,7.50000000000000,0.500000000000000;3092.63015600000,7.28836000000000,0.500000000000000;3346.17985100000,6.81216900000000,0.500000000000000;3596.34888400000,6.32275100000000,0.500000000000000;3427.31575400000,6.65343900000000,0.500000000000000;3677.48478700000,6.15079400000000,0.500000000000000;800,6.80000000000000,0.500000000000000;996.619337000000,7.07672000000000,0;996.619337000000,8.92857100000000,0;989.858012000000,10.7936510000000,0;6006.76132500000,10.7936510000000,0;1496.95740400000,10.7936510000000,0;2250.84516600000,10.7804230000000,0;3373.22515200000,10.7804230000000,0;4759.29682200000,10.5687830000000,0;5279.91886400000,10.8068780000000,0;4465.17917500000,10.8201060000000,0;6006.76132500000,0,0;6027.04530100000,9.27248700000000,0;6006.76132500000,7.42063500000000,0;6003.38066300000,5.48941800000000,0;6006.76132500000,3.45238100000000,0;6023.66463800000,1.74603200000000,0;4935.09127800000,0,0;5185.26031100000,-0.0264550000000000,0;5617.98512500000,-0.0264550000000000,0;800,7,0;800,9,0;800,11,0;1077.20232800000,7.01980400000000,0;1179.57005600000,7.04695000000000,0;1292.17455600000,6.97908400000000,0;1425.25260200000,6.96551100000000,0;1561.74290500000,7.08766900000000,0;1687.99643600000,7.27769300000000,0;1790.36416400000,7.46771700000000,0;1926.85446700000,7.63059400000000,0;2083.81831600000,7.80704500000000,0;2210.07184700000,7.90205700000000,0;2370.44795400000,7.96992200000000,0;2496.70148400000,7.99706900000000,0;2636.60404500000,7.95634900000000,0;2814.04144000000,7.83419100000000,0;2974.41754700000,7.76632500000000,0;3069.96075900000,7.63059400000000,0;3185.97751700000,7.30483900000000,0;3308.81879000000,7.10124200000000,0;3472.60715400000,6.84335300000000,0;3609.09745800000,6.54474400000000,0;3776.29808000000,6.25970900000000,0;3847.95548900000,5.73035700000000,0;3905.96386800000,5.32316300000000,0;3957.14773200000,5.01098100000000,0;4066.33997500000,4.65808000000000,0;4178.94447500000,4.41376400000000,0;4291.54897500000,4.22374000000000,0;4370.03090000000,3.76225400000000,0;4417.80250600000,3.26004800000000,0;4458.74959700000,2.75784200000000,0;4455.33734000000,2.24206300000000,0;4496.28443100000,1.87558900000000,0;4598.65215800000,1.57698000000000,0;4697.60762800000,1.38695700000000,0;4820.44890100000,1.14264000000000,0;4885.28179600000,0.708300000000000,0;4933.05340200000,0.341826000000000,0;1735.76804200000,9.62584400000000,0;3247.39815400000,9.46296600000000,0;4851.15922000000,9.27294200000000,0;5499.48816100000,9.02862600000000,0;1312.64810100000,8.43140900000000,0;1387.71776800000,8.15994600000000,0;1844.96028500000,8.55356700000000,0;1329.70938900000,9.85658700000000,0;2575.18340900000,8.94718700000000,0;2445.51762100000,9.80229400000000,0;3278.10847200000,8.47212800000000,0;3721.70195800000,7.34555900000000,0;4475.81088500000,5.05170000000000,0;4752.20375000000,3.72153400000000,0;5216.27078200000,1.78057700000000,0;5827.06489000000,1.14264000000000,0;5656.45201000000,1.40053000000000,0;5444.89204000000,5.18743200000000,0;4520.17023400000,7.64416700000000,0;4004.91933800000,9.69370900000000,0;4059.51545900000,7.67131400000000,0;5328.87528200000,7.22340000000000,0];

    map_data(:,3) = map_data(:,3) * abs(mod_DCP_CVVL);

    engine_out = engine_tech_mod( engine_out, mod_DCP_CVVL > 0, mod_DCP_CVVL < 0, map_data );
end

%% add or remove discrete cam phasing/discrete variable valve lift effectiveness
if mod_DCP_DVVL ~= 0
% see REVS_modify_engine() for original implementation
  	map_data = [1118.00404600000,-0.0137360000000000,9.50000000000000;1000,0.480769000000000,9.50000000000000;1064.05933900000,0.260989000000000,9.50000000000000;1000,0.961538000000000,9;1232.63654800000,-0.0137360000000000,9;1060.68779500000,0.686813000000000,9;1114.63250200000,0.412088000000000,9;1158.46257600000,0.260989000000000,9;1003.37154400000,1.51098900000000,8.50000000000000;1161.83412000000,1.15384600000000,8.50000000000000;1232.63654800000,0.989011000000000,8.50000000000000;1266.35198900000,0.645604000000000,8.50000000000000;1296.69588700000,0.329670000000000,8.50000000000000;1337.15441700000,0,8.50000000000000;1003.37154400000,2.07417600000000,8;1158.46257600000,2.23901100000000,8;1330.41132800000,2.40384600000000,8;1478.75927200000,2.55494500000000,8;1583.27714100000,2.33516500000000,8;1772.08361400000,1.99175800000000,8;1691.16655400000,2.17033000000000,8;1657.45111300000,1.53846200000000,8;1579.90559700000,1.22252700000000,8;1498.98853700000,0.879121000000000,8;1472.01618300000,0.549451000000000,8;1468.64463900000,0.233516000000000,8;1455.15846300000,-0.0137360000000000,8;1532.70397800000,2.44505500000000,8;1633.85030300000,2.25274700000000,8;1731.62508400000,2.08791200000000,8;1728.25354000000,1.82692300000000,8;1687.79501000000,1.66208800000000,8;1613.62103800000,1.35989000000000,8;1536.07552300000,1.04395600000000,8;1397.84221200000,2.44505500000000,8;1242.75118000000,2.29395600000000,8;1080.91706000000,2.14285700000000,8;800,1.90000000000000,8;1003.37154400000,2.65109900000000,7.50000000000000;1161.83412000000,2.85714300000000,7.50000000000000;1330.41132800000,3.06318700000000,7.50000000000000;1485.50236000000,3.20054900000000,7.50000000000000;1748.48280500000,2.89835200000000,7.50000000000000;1937.28927800000,2.50000000000000,7.50000000000000;2072.15104500000,2.23901100000000,7.50000000000000;2173.29737000000,1.99175800000000,7.50000000000000;2018.20633900000,1.23626400000000,7.50000000000000;1883.34457200000,0.576923000000000,7.50000000000000;1755.22589300000,0,7.50000000000000;800,2.50000000000000,7.50000000000000;1003.37154400000,3.22802200000000,7;1077.54551600000,3.29670300000000,7;1161.83412000000,3.32417600000000,7;1242.75118000000,3.39285700000000,7;1327.03978400000,3.46153800000000,7;1414.69993300000,3.53022000000000,7;1495.61699300000,3.55769200000000,7;1664.19420100000,3.42033000000000,7;1832.77140900000,3.28296700000000,7;2001.34861800000,3.13186800000000,7;2085.63722200000,3.07692300000000,7;2169.92582600000,2.92582400000000,7;2304.78759300000,2.59615400000000,7;2439.64935900000,2.28022000000000,7;2530.68105200000,2.00549500000000,7;2473.36480100000,1.42857100000000,7;2389.07619700000,0.934066000000000,7;2287.92987200000,0.439560000000000,7;2193.52663500000,0,7;800,3,7;1006.74308800000,3.80494500000000,6.50000000000000;1161.83412000000,3.84615400000000,6.50000000000000;1246.12272400000,3.90109900000000,6.50000000000000;1330.41132800000,3.88736300000000,6.50000000000000;1495.61699300000,3.95604400000000,6.50000000000000;1748.48280500000,3.79120900000000,6.50000000000000;1997.97707300000,3.64011000000000,6.50000000000000;2540.79568400000,2.98076900000000,6.50000000000000;2169.92582600000,3.43406600000000,6.50000000000000;2335.13149000000,3.22802200000000,6.50000000000000;2658.79973000000,2.40384600000000,6.50000000000000;2746.45987900000,2.00549500000000,6.50000000000000;2706.00134900000,1.57967000000000,6.50000000000000;2706.00134900000,1.40109900000000,6.50000000000000;2652.05664200000,0.906593000000000,6.50000000000000;2604.85502400000,0.412088000000000,6.50000000000000;2557.65340500000,-0.0137360000000000,6.50000000000000;1003.37154400000,4.28571400000000,6;1495.61699300000,4.35439600000000,6;1246.12272400000,4.32692300000000,6;1991.23398500000,4.17582400000000,6;1755.22589300000,4.24450500000000,6;2500.33715400000,3.58516500000000,6;2173.29737000000,3.95604400000000,6;2341.87457900000,3.76373600000000,6;2797.03304100000,2.99450500000000,6;2672.28590700000,3.24175800000000,6;2945.38098400000,2.01923100000000,6;2874.57855700000,2.43131900000000,6;2904.92245400000,1.40109900000000,6;2861.09238000000,0.906593000000000,6;2834.12002700000,0.576923000000000,6;2776.80377600000,-0.0274730000000000,6;800,4.20000000000000,6;1000,4.76648400000000,5.50000000000000;1495.61699300000,4.72527500000000,5.50000000000000;1246.12272400000,4.72527500000000,5.50000000000000;1981.11935300000,4.65659300000000,5.50000000000000;1751.85434900000,4.67033000000000,5.50000000000000;2510.45178700000,4.06593400000000,5.50000000000000;2995.95414700000,3.10439600000000,5.50000000000000;2166.55428200000,4.42307700000000,5.50000000000000;2335.13149000000,4.24450500000000,5.50000000000000;2675.65745100000,3.76373600000000,5.50000000000000;2847.60620400000,3.44780200000000,5.50000000000000;3093.72892800000,2.11538500000000,5.50000000000000;3063.38503000000,2.60989000000000,5.50000000000000;3107.21510500000,1.78571400000000,5.50000000000000;3110.58664900000,1.60714300000000,5.50000000000000;3090.35738400000,1.42857100000000,5.50000000000000;3097.10047200000,0.947802000000000,5.50000000000000;3043.15576500000,0.604396000000000,5.50000000000000;2989.21105900000,-0.0137360000000000,5.50000000000000;800,4.70000000000000,5.50000000000000;1000,5.09615400000000,5;1158.46257600000,5.09615400000000,5;1330.41132800000,5.05494500000000,5;1495.61699300000,5.06868100000000,5;1664.19420100000,5.06868100000000,5;1832.77140900000,5.05494500000000,5;1997.97707300000,5.06868100000000,5;2169.92582600000,4.89011000000000,5;2419.42009400000,4.65659300000000,5;2668.91436300000,4.36813200000000,5;2840.86311500000,4.16208800000000,5;2992.58260300000,3.96978000000000,5;3093.72892800000,3.47527500000000,5;3164.53135500000,3.13186800000000,5;3228.59069500000,2.47252700000000,5;3296.02157800000,1.84065900000000,5;3296.02157800000,1.31868100000000,5;3316.25084300000,1.07142900000000,5;3275.79231300000,0.645604000000000,5;3225.21915000000,0.315934000000000,5;3188.13216500000,-0.0137360000000000,5;800,5.10000000000000,5;1003.37154400000,5.31593400000000,4.50000000000000;1161.83412000000,5.28846200000000,4.50000000000000;1330.41132800000,5.26098900000000,4.50000000000000;1414.69993300000,5.21978000000000,4.50000000000000;1583.27714100000,5.21978000000000,4.50000000000000;1664.19420100000,5.24725300000000,4.50000000000000;1751.85434900000,5.24725300000000,4.50000000000000;1917.06001300000,5.28846200000000,4.50000000000000;2001.34861800000,5.28846200000000,4.50000000000000;2169.92582600000,5.19230800000000,4.50000000000000;2335.13149000000,5.12362600000000,4.50000000000000;2503.70869900000,5.06868100000000,4.50000000000000;2668.91436300000,4.84890100000000,4.50000000000000;2837.49157100000,4.65659300000000,4.50000000000000;3006.06878000000,4.46428600000000,4.50000000000000;3174.64598800000,3.79120900000000,4.50000000000000;3349.96628500000,2.98076900000000,4.50000000000000;3397.16790300000,2.47252700000000,4.50000000000000;3390.42481500000,2.30769200000000,4.50000000000000;3447.74106500000,1.81318700000000,4.50000000000000;3498.31422800000,1.31868100000000,4.50000000000000;3532.02967000000,0.975275000000000,4.50000000000000;3498.31422800000,0.645604000000000,4.50000000000000;3451.11261000000,0.315934000000000,4.50000000000000;3403.91099100000,-0.0137360000000000,4.50000000000000;1003.37154400000,5.60439600000000,4;1074.17397200000,5.56318700000000,4;1242.75118000000,5.46703300000000,4;1161.83412000000,5.53571400000000,4;1414.69993300000,5.39835200000000,4;1495.61699300000,5.35714300000000,4;1664.19420100000,5.42582400000000,4;1832.77140900000,5.49450500000000,4;1917.06001300000,5.49450500000000,4;1997.97707300000,5.53571400000000,4;2166.55428200000,5.42582400000000,4;2416.04855000000,5.32967000000000,4;2668.91436300000,5.19230800000000,4;2840.86311500000,5.09615400000000,4;2989.21105900000,4.95879100000000,4;3491.57114000000,2.98076900000000,4;3120.70128100000,4.46428600000000,4;3289.27849000000,3.79120900000000,4;3380.31018200000,3.46153800000000,4;3548.88739000000,2.47252700000000,4;3542.14430200000,2.29395600000000,4;3596.08900900000,1.81318700000000,4;3680.37761300000,0.975275000000000,4;3643.29062700000,1.31868100000000,4;3646.66217100000,0.467033000000000,4;3629.80445000000,0.151099000000000,4;3616.31827400000,0,4;1003.37154400000,5.79670300000000,3.50000000000000;1161.83412000000,5.72802200000000,3.50000000000000;1495.61699300000,5.53571400000000,3.50000000000000;1347.26904900000,5.60439600000000,3.50000000000000;1987.86244100000,5.76923100000000,3.50000000000000;1579.90559700000,5.56318700000000,3.50000000000000;1755.22589300000,5.64560400000000,3.50000000000000;2493.59406600000,5.56318700000000,3.50000000000000;2257.58597400000,5.63186800000000,3.50000000000000;2995.95414700000,5.26098900000000,3.50000000000000;2668.91436300000,5.45329700000000,3.50000000000000;2837.49157100000,5.34340700000000,3.50000000000000;3147.67363500000,4.95879100000000,3.50000000000000;3616.31827400000,3.03571400000000,3.50000000000000;3285.90694500000,4.39560400000000,3.50000000000000;3444.36952100000,3.72252700000000,3.50000000000000;3828.72555600000,1.07142900000000,3.50000000000000;3690.49224500000,2.25274700000000,3.50000000000000;3768.03776100000,1.57967000000000,3.50000000000000;3848.95482100000,-0.0137360000000000,3.50000000000000;3852.32636500000,0.439560000000000,3.50000000000000;3828.72555600000,0.535714000000000,3.50000000000000;3838.84018900000,0.755495000000000,3.50000000000000;1003.37154400000,6.03022000000000,3;1077.54551600000,6.03022000000000,3;1161.83412000000,5.96153800000000,3;1327.03978400000,5.81044000000000,3;1414.69993300000,5.75549500000000,3;2001.34861800000,6.01648400000000,3;1498.98853700000,5.64560400000000,3;1708.02427500000,5.82417600000000,3;1832.77140900000,5.92033000000000,3;2995.95414700000,5.50824200000000,3;2254.21443000000,5.90659300000000,3;2584.62575900000,5.71428600000000,3;3262.30613600000,4.97252700000000,3;3144.30209000000,5.20604400000000,3;3734.32232000000,3.10439600000000,3;3430.88334500000,4.29945100000000,3;3592.71746500000,3.65384600000000,3;3815.23938000000,2.48626400000000,3;3896.15644000000,1.79945100000000,3;3997.30276500000,1.04395600000000,3;4034.38975100000,0.480769000000000,3;4071.47673600000,0,3;800,6.10000000000000,3;1984.49089700000,6.26373600000000,2.50000000000000;1498.98853700000,5.86538500000000,2.50000000000000;1232.63654800000,6.09890100000000,2.50000000000000;1091.03169300000,6.11263700000000,2.50000000000000;1003.37154400000,6.20879100000000,2.50000000000000;1330.41132800000,5.98901100000000,2.50000000000000;1620.36412700000,5.93406600000000,2.50000000000000;1748.48280500000,6.03022000000000,2.50000000000000;2507.08024300000,6.08516500000000,2.50000000000000;2193.52663500000,6.16758200000000,2.50000000000000;2345.24612300000,6.11263700000000,2.50000000000000;2995.95414700000,5.75549500000000,2.50000000000000;2722.85906900000,5.90659300000000,2.50000000000000;2864.46392400000,5.82417600000000,2.50000000000000;3383.68172600000,5,2.50000000000000;3124.07282500000,5.50824200000000,2.50000000000000;3258.93459200000,5.21978000000000,2.50000000000000;3869.18408600000,3.04945100000000,2.50000000000000;3508.42886000000,4.45054900000000,2.50000000000000;3636.54753900000,3.94230800000000,2.50000000000000;3747.80849600000,3.50274700000000,2.50000000000000;4125.42144300000,0.961538000000000,2.50000000000000;3943.35805800000,2.43131900000000,2.50000000000000;4007.41739700000,1.93681300000000,2.50000000000000;4071.47673600000,1.40109900000000,2.50000000000000;4290.62710700000,-0.0137360000000000,2.50000000000000;4189.48078200000,0.576923000000000,2.50000000000000;4243.42548900000,0.260989000000000,2.50000000000000;1006.74308800000,6.37362600000000,2;1465.27309500000,6.04395600000000,2;1246.12272400000,6.18131900000000,2;1981.11935300000,6.52472500000000,2;1674.30883300000,6.20879100000000,2;1826.02832100000,6.35989000000000,2;2486.85097800000,6.57967000000000,2;2173.29737000000,6.55219800000000,2;2335.13149000000,6.55219800000000,2;2995.95414700000,6.03022000000000,2;2675.65745100000,6.38736300000000,2;2840.86311500000,6.18131900000000,2;3515.17194900000,4.97252700000000,2;3181.38907600000,5.65934100000000,2;3329.73702000000,5.32967000000000,2;4004.04585300000,3.03571400000000,2;3697.23533400000,4.24450500000000,2;3875.92717500000,3.53022000000000,2;4051.24747100000,2.58241800000000,2;4084.96291300000,2.30769200000000,2;4101.82063400000,2.08791200000000,2;4155.76534100000,1.64835200000000,2;4223.19622400000,1.20879100000000,2;4253.54012100000,1.00274700000000,2;4506.40593400000,-0.0274730000000000,2;4354.68644600000,0.604396000000000,2;4435.60350600000,0.274725000000000,2;1000,6.49725300000000,1.50000000000000;1165.20566400000,6.41483500000000,1.50000000000000;1333.78287300000,6.31868100000000,1.50000000000000;1414.69993300000,6.27747300000000,1.50000000000000;1498.98853700000,6.27747300000000,1.50000000000000;1667.56574500000,6.44230800000000,1.50000000000000;1917.06001300000,6.67582400000000,1.50000000000000;2048.55023600000,6.78571400000000,1.50000000000000;2490.22252200000,7.04670300000000,1.50000000000000;2254.21443000000,6.90934100000000,1.50000000000000;2557.65340500000,7.03296700000000,1.50000000000000;2995.95414700000,6.52472500000000,1.50000000000000;2712.74443700000,6.84065900000000,1.50000000000000;2874.57855700000,6.64835200000000,1.50000000000000;3494.94268400000,5.49450500000000,1.50000000000000;3178.01753200000,6.15384600000000,1.50000000000000;3349.96628500000,5.82417600000000,1.50000000000000;3747.80849600000,4.49175800000000,1.50000000000000;3602.83209700000,5.09615400000000,1.50000000000000;3666.89143600000,4.80769200000000,1.50000000000000;4118.67835500000,3.02197800000000,1.50000000000000;3872.55563000000,4.01098900000000,1.50000000000000;3997.30276500000,3.50274700000000,1.50000000000000;4155.76534100000,2.52747300000000,1.50000000000000;4196.22387100000,2.00549500000000,1.50000000000000;4280.51247500000,1.52472500000000,1.50000000000000;4368.17262300000,1.01648400000000,1.50000000000000;4455.83277100000,0.686813000000000,1.50000000000000;4533.37828700000,0.357143000000000,1.50000000000000;4617.66689100000,-0.0137360000000000,1.50000000000000;1003.37154400000,6.66208800000000,1;1165.20566400000,6.59340700000000,1;1256.23735700000,6.62087900000000,1;1330.41132800000,6.57967000000000,1;1414.69993300000,6.55219800000000,1;1498.98853700000,6.53846200000000,1;1994.60552900000,7.04670300000000,1;1667.56574500000,6.70329700000000,1;1836.14295300000,6.88186800000000,1;2085.63722200000,7.07417600000000,1;2254.21443000000,7.17033000000000,1;2496.96561000000,7.34890100000000,1;2382.33310900000,7.26648400000000,1;3002.69723500000,7.01923100000000,1;2672.28590700000,7.22527500000000,1;2840.86311500000,7.10164800000000,1;3498.31422800000,6.03022000000000,1;3178.01753200000,6.64835200000000,1;3349.96628500000,6.31868100000000,1;3670.26298000000,5.32967000000000,1;3838.84018900000,4.67033000000000,1;4007.41739700000,4.01098900000000,1;4122.04989900000,3.50274700000000,1;4260.28321000000,3.02197800000000,1;4283.88401900000,2.48626400000000,1;4287.25556300000,2.17033000000000,1;4334.45718100000,1.82692300000000,1;4438.97505100000,1.33241800000000,1;4540.12137600000,0.837912000000000,1;4674.98314200000,0.384615000000000,1;4705.32704000000,0.178571000000000,1;4759.27174600000,-0.0137360000000000,1;1003.37154400000,6.84065900000000,0.500000000000000;1080.91706000000,6.81318700000000,0.500000000000000;1161.83412000000,6.81318700000000,0.500000000000000;1330.41132800000,6.77197800000000,0.500000000000000;1418.07147700000,6.74450500000000,0.500000000000000;1498.98853700000,6.77197800000000,0.500000000000000;1667.56574500000,6.95054900000000,0.500000000000000;1991.23398500000,7.51373600000000,0.500000000000000;1832.77140900000,7.21153800000000,0.500000000000000;2493.59406600000,7.69230800000000,0.500000000000000;2169.92582600000,7.54120900000000,0.500000000000000;2335.13149000000,7.60989000000000,0.500000000000000;2995.95414700000,7.50000000000000,0.500000000000000;2672.28590700000,7.60989000000000,0.500000000000000;2840.86311500000,7.54120900000000,0.500000000000000;3761.29467300000,5.98901100000000,0.500000000000000;3262.30613600000,6.97802200000000,0.500000000000000;3596.08900900000,6.31868100000000,0.500000000000000;3427.51180000000,6.64835200000000,0.500000000000000;3815.23938000000,5.49450500000000,0.500000000000000;3875.92717500000,5.01373600000000,0.500000000000000;3963.58732300000,4.64285700000000,0.500000000000000;4081.59136900000,4.32692300000000,0.500000000000000;4165.87997300000,4.16208800000000,0.500000000000000;4253.54012100000,3.99725300000000,0.500000000000000;4020.90357400000,4.47802200000000,0.500000000000000;4297.37019600000,3.69505500000000,0.500000000000000;4334.45718100000,3.32417600000000,0.500000000000000;4374.91571100000,3.06318700000000,0.500000000000000;4371.54416700000,2.82967000000000,0.500000000000000;4388.40188800000,2.50000000000000,0.500000000000000;4381.65880000000,2.32142900000000,0.500000000000000;4415.37424100000,2.22527500000000,0.500000000000000;4408.63115300000,2.00549500000000,0.500000000000000;4469.31894800000,1.71703300000000,0.500000000000000;4499.66284600000,1.51098900000000,0.500000000000000;4755.90020200000,1.00274700000000,0.500000000000000;4600.80917100000,1.33241800000000,0.500000000000000;4671.61159800000,1.19505500000000,0.500000000000000;4803.10182100000,0.659341000000000,0.500000000000000;4840.18880600000,0.357143000000000,0.500000000000000;4877.27579200000,-0.0137360000000000,0.500000000000000;800,6.90000000000000,0.500000000000000;1000,7.07417600000000,0;1003.37154400000,10.8241760000000,0;1003.37154400000,10.0137360000000,0;1006.74308800000,8.68131900000000,0;1583.27714100000,10.8241760000000,0;2729.60215800000,10.8379120000000,0;4000.67430900000,10.8241760000000,0;5211.05866500000,10.8379120000000,0;6003.37154400000,10.8241760000000,0;6000,9.02472500000000,0;6003.37154400000,7.23901100000000,0;6000,5.20604400000000,0;6003.37154400000,3.11813200000000,0;6003.37154400000,1.35989000000000,0;6003.37154400000,-0.0274730000000000,0;5676.33176000000,-0.0274730000000000,0;5265.00337200000,-0.0274730000000000,0;4954.82130800000,-0.0274730000000000,0;4927.84895500000,0.412088000000000,0;4833.44571800000,1.07142900000000,0;4715.44167200000,1.51098900000000,0;4540.12137600000,1.90934100000000,0;4465.94740400000,2.99450500000000,0;4354.68644600000,4.16208800000000,0;4105.19217800000,4.78022000000000,0;3966.95886700000,5.05494500000000,0;3855.69791000000,5.85164800000000,0;3687.12070100000,6.63461500000000,0;3424.14025600000,7.07417600000000,0;3154.41672300000,7.54120900000000,0;2847.60620400000,7.92582400000000,0;2544.16722900000,8.03571400000000,0;2244.09979800000,8.00824200000000,0;1987.86244100000,7.92582400000000,0;1792.31287900000,7.70604400000000,0;1654.07956800000,7.40384600000000,0;1498.98853700000,7.18406600000000,0;1380.98449100000,7.14285700000000,0;1246.12272400000,7.06044000000000,0;1111.26095800000,7.10164800000000,0;1549.56169900000,9.21703300000000,0;2712.74443700000,9.27197800000000,0;3869.18408600000,9.20329700000000,0;5497.63991900000,5.15109900000000,0;5207.68712100000,8.76373600000000,0;4712.07012800000,6.70329700000000,0;5028.99528000000,4.03846200000000,0;5581.92852300000,2.56868100000000,0;4155.76534100000,7.03296700000000,0;800,7.25000000000000,0;800,9,0;800,11,0];

    map_data(:,3) = map_data(:,3) * abs(mod_DCP_DVVL);

    engine_out = engine_tech_mod( engine_out, mod_DCP_DVVL > 0, mod_DCP_DVVL < 0, map_data );
end

%% add or remove discrete cam phasing
if mod_DCP ~= 0 
% see REVS_modify_engine() for original implementation
	map_data = [1626.17355100000,2.72921500000000,6.50000000000000;2365.97196600000,2.69907600000000,6.50000000000000;2435.52566300000,2.15658800000000,6.50000000000000;2157.31087500000,2.32234800000000,6.50000000000000;1632.49661500000,2.33741700000000,6.50000000000000;2011.88041700000,2.69907600000000,6.50000000000000;0,2.01526700000000,6;1470.08547000000,3.00763400000000,6;2484.07148400000,3.00763400000000,6;2651.12665100000,2.01526700000000,6;2515.15151500000,1.83206100000000,6;1975.13597500000,2.01526700000000,6;2541.61810400000,2.67438000000000,6;2602.80691500000,2.34899800000000,6;1984.12004700000,3.01455200000000,6;1722.36791000000,3.01455200000000,6;2228.87529100000,3.01455200000000,6;2126.89393900000,1.93487600000000,6;2347.85353500000,1.86092600000000,6;1613.58780100000,2.02361600000000,6;1800.55361300000,2.02361600000000,6;1470.81390800000,2.52647900000000,6;1477.85547800000,3.51145000000000,5.50000000000000;2484.07148400000,3.51145000000000,5.50000000000000;2740.48174000000,3.00763400000000,5.50000000000000;2825.95182600000,2.03053400000000,5.50000000000000;2484.07148400000,1.63358800000000,5.50000000000000;1986.79098700000,1.75572500000000,5.50000000000000;1477.85547800000,1.75572500000000,5.50000000000000;1357.42035700000,2.03053400000000,5.50000000000000;1357.42035700000,2.91603100000000,5.50000000000000;1708.77039600000,3.51741400000000,5.50000000000000;1977.32129000000,3.50262400000000,5.50000000000000;2228.87529100000,3.51741400000000,5.50000000000000;2623.20318600000,3.25119300000000,5.50000000000000;2789.77272700000,2.46731900000000,5.50000000000000;2650.39821300000,1.80176500000000,5.50000000000000;2232.27467000000,1.68344500000000,5.50000000000000;1729.16666700000,1.75739500000000,5.50000000000000;1473.97047400000,4,5;2985.23698500000,4,5;2985.23698500000,2.01526700000000,5;2526.80652700000,1.43511500000000,5;2060.60606100000,1.48091600000000,5;1815.85081600000,1.52671800000000,5;1477.85547800000,1.51145000000000,5;1225.33022500000,2,5;1225.33022500000,3.00763400000000,5;1334.83877200000,3.45825400000000,5;1219.25990700000,2.52647900000000,5;1345.03690800000,1.75739500000000,5;2317.25913000000,1.46159400000000,5;2820.36713300000,1.80176500000000,5;2657.19697000000,1.60949400000000,5;2990.33605300000,3.38430300000000,5;2990.33605300000,2.67438000000000,5;2691.19075400000,4.00548700000000,5;2398.84421100000,4.00548700000000,5;2065.70512800000,4.00548700000000,5;1807.35237000000,4.00548700000000,5;1225.33022500000,4,4.50000000000000;1477.85547800000,4.48855000000000,4.50000000000000;1982.90598300000,4.97709900000000,4.50000000000000;2452.99145300000,4.50381700000000,4.50000000000000;2985.23698500000,4.48855000000000,4.50000000000000;3121.21212100000,3.98473300000000,4.50000000000000;3241.64724200000,3.02290100000000,4.50000000000000;3245.53224600000,2,4.50000000000000;2985.23698500000,1.49618300000000,4.50000000000000;2557.88655800000,1.22137400000000,4.50000000000000;1982.90598300000,1.23664100000000,4.50000000000000;1477.85547800000,1.25190800000000,4.50000000000000;1108.78010900000,2,4.50000000000000;1108.78010900000,3.06870200000000,4.50000000000000;1103.68104100000,2.51168900000000,4.50000000000000;1239.65617700000,1.74260500000000,4.50000000000000;1732.56604500000,1.25453200000000,4.50000000000000;2279.86596700000,1.23974200000000,4.50000000000000;2786.37334900000,1.37285300000000,4.50000000000000;2728.58391600000,4.50834900000000,4.50000000000000;2235.67404800000,4.74499000000000,4.50000000000000;1742.76418000000,4.73020000000000,4.50000000000000;3116.11305400000,1.74260500000000,4.50000000000000;1171.66860900000,3.54699400000000,4.50000000000000;3181.64714100000,3.49774100000000,4.50000000000000;3244.87777500000,2.45797000000000,4.50000000000000;1000,4.99236600000000,4;1528.36052800000,4.99236600000000,4;1994.56099500000,5.32824400000000,4;2433.56643400000,4.99236600000000,4;2981.35198100000,4.99236600000000,4;3490.28749000000,3.00763400000000,4;3490.28749000000,1.02290100000000,4;1003.88500400000,1.02290100000000,4;1154.67171700000,4.99642200000000,4;1331.43939400000,4.99642200000000,4;1232.85742000000,1.01789100000000,4;1464.01515200000,1.00310100000000,4;1647.58158500000,1.01789100000000,4;1905.93434300000,1.01789100000000,4;2174.48523700000,1.01789100000000,4;2426.03923900000,1.01789100000000,4;2630.00194300000,1.01789100000000,4;2850.96153800000,1.00310100000000,4;3054.92424200000,1.00310100000000,4;3262.28632500000,1.00310100000000,4;3490.04467800000,1.62428400000000,4;3490.04467800000,2.33420800000000,4;3381.26456900000,3.44346400000000,4;3207.89627000000,4.13859700000000,4;2789.77272700000,4.99642200000000,4;2616.40442900000,4.99642200000000,4;2222.07653500000,5.15911300000000,4;1831.14801900000,5.21827300000000,4;1664.57847700000,5.09995200000000,4;987.544151000000,4.31147400000000,4;993.867214000000,3.54294800000000,4;993.867214000000,2.68400700000000,4;993.867214000000,1.77985900000000,4;993.867214000000,4.91423900000000,4;993.867214000000,1.13188600000000,4;993.867214000000,4.98958500000000,4;993.867214000000,1.01133300000000,4;993.867214000000,4.85396300000000,4;993.867214000000,4.91423900000000,4;993.867214000000,2.08124200000000,4;993.867214000000,1.91548100000000,4;1000,5.26717600000000,3.50000000000000;1501.16550100000,5.25190800000000,3.50000000000000;1986.79098700000,5.66412200000000,3.50000000000000;2449.10644900000,5.26717600000000,3.50000000000000;3000.77700100000,5.25190800000000,3.50000000000000;3606.83760700000,3.03816800000000,3.50000000000000;3630.14763000000,2.54961800000000,3.50000000000000;3661.22766100000,2.10687000000000,3.50000000000000;3668.99766900000,1.89313000000000,3.50000000000000;3742.81274300000,1.05343500000000,3.50000000000000;3509.71251000000,0.763359000000000,3.50000000000000;1000,0.748092000000000,3.50000000000000;3143.70876100000,4.91423900000000,3.50000000000000;3377.66210600000,3.99502200000000,3.50000000000000;2897.10928900000,0.755158000000000,3.50000000000000;2214.21844500000,0.755158000000000,3.50000000000000;1645.14274100000,0.755158000000000,3.50000000000000;1151.94379800000,5.26082900000000,3.50000000000000;1328.98957300000,5.26082900000000,3.50000000000000;1177.23605200000,0.740089000000000,3.50000000000000;1423.83552300000,0.740089000000000,3.50000000000000;1000,5.49618300000000,3;1458.43045800000,5.49618300000000,3;1975.13597500000,6.03053400000000,3;2484.07148400000,5.49618300000000,3;2996.89199700000,5.49618300000000,3;3249.41724900000,4.97709900000000,3;3746.69774700000,3.03816800000000,3;3781.66278200000,2.68702300000000,3;3797.20279700000,2.27480900000000,3;3983.68298400000,1.03816800000000,3;3521.36752100000,0.519084000000000,3;992.229992000000,0.519084000000000,3;3504.12337300000,3.94981500000000,3;3889.83023900000,1.64423700000000,3;3763.36897200000,0.785296000000000,3;2972.98605000000,0.514052000000000,3;2384.94115600000,0.514052000000000,3;1904.38834000000,0.514052000000000,3;1449.12777700000,0.514052000000000,3;1158.26686200000,5.48686600000000,3;1328.98957300000,5.48686600000000,3;1739.98869200000,5.77318000000000,3;2245.83376200000,5.72797200000000,3;2751.67883200000,5.48686600000000,3;1000,5.77099200000000,2.50000000000000;1485.62548600000,5.78626000000000,2.50000000000000;1998.44599800000,6.25954200000000,2.50000000000000;2484.07148400000,5.75572500000000,2.50000000000000;2993.00699300000,5.75572500000000,2.50000000000000;3365.96736600000,5.00763400000000,2.50000000000000;3855.47785500000,3.08396900000000,2.50000000000000;4111.88811200000,1.03816800000000,2.50000000000000;3991.45299100000,0.488550000000000,2.50000000000000;3630.14763000000,0.290076000000000,2.50000000000000;996.114996000000,0.259542000000000,2.50000000000000;1240.46668600000,5.75811100000000,2.50000000000000;1746.31175500000,6.01428600000000,2.50000000000000;2245.83376200000,5.98414800000000,2.50000000000000;2751.67883200000,5.75811100000000,2.50000000000000;3561.03094400000,4.23612800000000,2.50000000000000;3940.41474600000,2.39769400000000,2.50000000000000;4022.61457000000,1.73465200000000,2.50000000000000;3769.69203500000,0.408568000000000,2.50000000000000;992.229992000000,6.03053400000000,2;1505.05050500000,6.03053400000000,2;1986.79098700000,6.51908400000000,2;2476.30147600000,6.03053400000000,2;2989.12198900000,6.01526700000000,2;3494.17249400000,4.97709900000000,2;3999.22299900000,3.00763400000000,2;4243.97824400000,0.992366000000000,2;4500.38850000000,-0.0152670000000000,2;1158.26686200000,6.02935500000000,2;1752.63481900000,6.30060000000000,2;2245.83376200000,6.25539200000000,2;2720.06351500000,6.02935500000000,2;3244.87777500000,5.51700500000000,2;3662.19995800000,4.32654300000000,2;3839.24573200000,3.63336300000000,2;4079.52214000000,2.15658800000000,2;4180.69115400000,1.52368400000000,2;4370.38305600000,0.514052000000000,2;4433.61368900000,0.272946000000000,2;3086.80119100000,5.81838700000000,2;2840.20171900000,6.02935500000000,2;4313.47548500000,0.755158000000000,2;4472.46503500000,0.130487000000000,2;992.229992000000,6.25954200000000,1.50000000000000;1497.28049700000,6.25954200000000,1.50000000000000;1979.02097900000,6.76335900000000,1.50000000000000;2437.45143700000,6.53435100000000,1.50000000000000;2985.23698500000,6.51908400000000,1.50000000000000;3513.59751400000,5.40458000000000,1.50000000000000;4115.77311600000,2.99236600000000,1.50000000000000;4212.89821300000,1.92366400000000,1.50000000000000;4376.06837600000,0.992366000000000,1.50000000000000;4616.93861700000,-0.0152670000000000,1.50000000000000;1161.47047400000,6.25357800000000,1.50000000000000;1331.43939400000,6.25357800000000,1.50000000000000;1746.16355900000,6.50501000000000,1.50000000000000;2171.08585900000,6.65291000000000,1.50000000000000;2670.79448300000,6.51980000000000,1.50000000000000;3252.08819000000,6.00214700000000,1.50000000000000;3728.00116600000,4.56751000000000,1.50000000000000;3925.16511300000,3.78363500000000,1.50000000000000;4166.52097900000,2.40815800000000,1.50000000000000;4309.29487200000,1.35806300000000,1.50000000000000;4482.66317000000,0.574189000000000,1.50000000000000;4557.44949500000,0.278387000000000,1.50000000000000;4431.67249400000,0.781250000000000,1.50000000000000;988.344988000000,6.50381700000000,1;1501.16550100000,6.53435100000000,1;1994.56099500000,7.02290100000000,1;2985.23698500000,7.02290100000000,1;3509.71251000000,6,1;4240.09324000000,3.05343500000000,1;4298.36829800000,1.98473300000000,1;4756.79875700000,-0.0305340000000000,1;1158.07109600000,6.51980000000000,1;1375.63131300000,6.51980000000000,1;1746.16355900000,6.78602100000000,1;2249.27156200000,7.02266200000000,1;2670.79448300000,7.03745200000000,1;3173.90248600000,6.65291000000000,1;3377.86519000000,6.25357800000000,1;3660.01359800000,5.33659400000000,1;3836.78127400000,4.64146000000000,1;4044.14335700000,3.84279600000000,1;4275.30108800000,2.54126900000000,1;4411.27622400000,1.46159400000000,1;4526.85508900000,0.899571000000000,1;4669.62898200000,0.337548000000000,1;992.229992000000,6.76335900000000,0.500000000000000;1505.05050500000,6.76335900000000,0.500000000000000;1745.92074600000,7.05343500000000,0.500000000000000;1979.02097900000,7.48091600000000,0.500000000000000;2993.00699300000,7.48091600000000,0.500000000000000;3754.46775400000,6,0.500000000000000;3863.24786300000,5.03816800000000,0.500000000000000;4003.10800300000,4.50381700000000,0.500000000000000;4251.74825200000,3.98473300000000,0.500000000000000;4364.41336400000,3.09923700000000,0.500000000000000;4399.37839900000,2.01526700000000,0.500000000000000;4527.58352800000,1.45038200000000,0.500000000000000;4764.56876500000,0.992366000000000,0.500000000000000;4873.34887300000,-0.0152670000000000,0.500000000000000;1246.78974900000,6.75267400000000,0.500000000000000;2372.29502900000,7.49106100000000,0.500000000000000;2675.80207100000,7.47599200000000,0.500000000000000;3289.13921900000,6.93350300000000,0.500000000000000;3510.44643700000,6.49649800000000,0.500000000000000;4819.32055500000,0.468845000000000,0.500000000000000;4383.02918200000,2.66893800000000,0.500000000000000;1117.27855500000,6.75644100000000,0.500000000000000;1409.62509700000,6.74165100000000,0.500000000000000;1623.78593600000,6.88955200000000,0.500000000000000;1831.14801900000,7.21493300000000,0.500000000000000;2171.08585900000,7.46636500000000,0.500000000000000;3170.50310800000,7.15577300000000,0.500000000000000;3809.58624700000,5.48449400000000,0.500000000000000;3632.81857000000,6.23878800000000,0.500000000000000;4316.09362900000,3.50262400000000,0.500000000000000;4159.72222200000,4.15338700000000,0.500000000000000;4081.53651900000,4.33086800000000,0.500000000000000;4472.46503500000,1.71302500000000,0.500000000000000;4384.08119700000,2.33420800000000,0.500000000000000;4666.22960400000,1.18058200000000,0.500000000000000;4601.64141400000,1.32848300000000,0.500000000000000;4792.00660500000,0.707300000000000,0.500000000000000;4846.39665900000,0.219227000000000,0.500000000000000;986.787322000000,7.32530100000000,0;1284.64983700000,7.31023200000000,0;1437.04461200000,7.29516200000000,0;1818.03154900000,7.89792800000000,0;2192.09145200000,8.01848100000000,0;3023.33567900000,7.98834300000000,0;3480.52000400000,7.18967800000000,0;3972.33950500000,6.13483900000000,0;4062.39096300000,4.91423900000000,0;4484.94011200000,4.19092100000000,0;4574.99157000000,3.06073600000000,0;4665.04302800000,2.03603400000000,0;4983.68664900000,1.22230100000000,0;5115.30031800000,-0.0284370000000000,0;6008.88786300000,-0.0284370000000000,0;6008.88786300000,11.1829990000000,0;986.787322000000,11.1829990000000,0;1645.37357800000,7.42152900000000,0;800,7.10000000000000,0;800,9,0;800,11,0];

    map_data(:,3) = map_data(:,3) * abs(mod_DCP);

    engine_out = engine_tech_mod( engine_out, mod_DCP > 0, mod_DCP < 0, map_data );
end

%% add or remove continuous cam phasing
if mod_CCP ~= 0
% see REVS_modify_engine() for original implementation
	map_data = [1547.39428700000,3.77186600000000,5.50000000000000;1533.83692200000,2.32486700000000,5.50000000000000;1960.89390700000,2.32486700000000,5.50000000000000;2452.34837400000,1.95573500000000,5.50000000000000;2608.25806700000,2.13291800000000,5.50000000000000;2462.51639800000,2.75306100000000,5.50000000000000;1981.22995400000,3.71280500000000,5.50000000000000;1472.82878200000,4.02287600000000,5;1988.00863700000,4.03764100000000,5;2499.79915000000,3.01883600000000,5;2740.44237200000,2.05909200000000,5;2486.24178600000,1.71949000000000,5;1977.84061300000,2.04432700000000,5;1476.21812300000,2.05909200000000,5;1472.82878200000,3.38796800000000,5;1472.82878200000,2.67923400000000,5;1611.79176900000,4.03764100000000,5;1815.15223800000,4.03764100000000,5;2174.42240000000,3.65374400000000,5;2343.88945700000,3.31414200000000,5;2625.20477300000,2.53158100000000,5;2604.86872600000,1.85237800000000,5;2367.61484500000,1.76378600000000,5;2177.81174100000,1.89667400000000,5;1642.29583900000,2.04432700000000,5;1801.59487300000,2.05909200000000,5;1228.79621900000,4.02287600000000,4.50000000000000;1482.99680500000,4.51013100000000,4.50000000000000;1984.61929500000,5.01215100000000,4.50000000000000;2482.85244400000,4.02287600000000,4.50000000000000;2994.64295800000,4.02287600000000,4.50000000000000;2994.64295800000,2.05909200000000,4.50000000000000;2493.02046800000,1.36512300000000,4.50000000000000;1984.61929500000,1.54230700000000,4.50000000000000;1482.99680500000,1.54230700000000,4.50000000000000;1228.79621900000,2.02956100000000,4.50000000000000;1225.40687800000,3.35843800000000,4.50000000000000;1222.01753700000,2.67923400000000,4.50000000000000;1354.20184200000,1.76378600000000,4.50000000000000;1727.02936800000,1.54230700000000,4.50000000000000;2232.04119900000,1.42418400000000,4.50000000000000;2662.48752500000,1.60136800000000,4.50000000000000;2821.78655900000,1.77855100000000,4.50000000000000;2994.64295800000,2.64970400000000,4.50000000000000;2998.03229900000,3.40273400000000,4.50000000000000;2642.15147800000,4.02287600000000,4.50000000000000;2825.17590100000,4.02287600000000,4.50000000000000;2323.55341000000,4.36247800000000,4.50000000000000;2150.69701200000,4.70207900000000,4.50000000000000;1652.46386300000,4.68731400000000,4.50000000000000;1815.15223800000,4.83496700000000,4.50000000000000;994.931680000000,4.03764100000000,4;1482.99680500000,5.01215100000000,4;1981.22995400000,5.32222200000000,4;2252.37724600000,4.98262000000000,4;2479.46310300000,4.52489600000000,4;2991.25361700000,4.51013100000000,4;3177.66738000000,3.91951900000000,4;3245.45420300000,3.10742800000000,4;3245.45420300000,2.00003100000000,4;2984.47493500000,1.04028700000000,4;1466.05009900000,1.05505200000000,4;1005.09970300000,2.00003100000000,4;1171.17741900000,4.39200800000000,4;1323.69777100000,4.71684500000000,4;1662.63188600000,5.14503800000000,4;1832.09894400000,5.23363000000000,4;2167.64371700000,5.11550800000000,4;2638.76213700000,4.52489600000000,4;2825.17590100000,4.52489600000000,4;3092.93385100000,4.22959000000000,4;3211.56079200000,3.50609100000000,4;3248.84354400000,2.53158100000000,4;3160.72067400000,1.67519400000000,4;3082.76582800000,1.36512300000000,4;1350.81250000000,1.27653100000000,4;1194.90280800000,1.60136800000000,4;1716.86134500000,1.04028700000000,4;2089.68887100000,1.04028700000000,4;2425.23364500000,1.04028700000000,4;2747.22105400000,1.04028700000000,4;1001.71036200000,5.02691600000000,3.50000000000000;1191.51346600000,5.10074200000000,3.50000000000000;1496.55417000000,5.26316100000000,3.50000000000000;1981.22995400000,5.66182400000000,3.50000000000000;2469.29508000000,5.02691600000000,3.50000000000000;2987.86427600000,5.02691600000000,3.50000000000000;3347.13443800000,3.94904900000000,3.50000000000000;3492.87610700000,3.03360100000000,3.50000000000000;3492.87610700000,1.02552100000000,3.50000000000000;3055.65109900000,0.804042000000000,3.50000000000000;1466.05009900000,0.774511000000000,3.50000000000000;1001.71036200000,0.995991000000000,3.50000000000000;1750.75475600000,5.48464000000000,3.50000000000000;2167.64371700000,5.44034400000000,3.50000000000000;2306.60670500000,5.24839500000000,3.50000000000000;2638.76213700000,5.02691600000000,3.50000000000000;2808.22919500000,5.02691600000000,3.50000000000000;3109.88055700000,4.68731400000000,3.50000000000000;3238.67552100000,4.30341700000000,3.50000000000000;3418.31060200000,3.49132500000000,3.50000000000000;3496.26544800000,2.39869400000000,3.50000000000000;3499.65478900000,1.68996000000000,3.50000000000000;3340.35575500000,0.951695000000000,3.50000000000000;3191.22474500000,0.848338000000000,3.50000000000000;2801.45051200000,0.774511000000000,3.50000000000000;2252.37724600000,0.774511000000000,3.50000000000000;1832.09894400000,0.774511000000000,3.50000000000000;1320.30843000000,0.848338000000000,3.50000000000000;1164.39873700000,0.922164000000000,3.50000000000000;1005.09970300000,5.35175200000000,3;1415.20998200000,5.46987500000000,3;1984.61929500000,6.04572100000000,3;2472.68442100000,5.35175200000000,3;2991.25361700000,5.35175200000000,3;3275.95827300000,4.64301800000000,3;3516.60149500000,3.94904900000000,3;3645.39645900000,3.15172400000000,3;3713.18328200000,2.50205100000000,3;3730.12998800000,2.14768400000000,3;3997.88793800000,1.04028700000000,3;3391.19587300000,0.656389000000000,3;2974.30691100000,0.553032000000000,3;1550.78362800000,0.553032000000000,3;1462.66075800000,0.553032000000000,3;1005.09970300000,0.685920000000000,3;800,0.800000000000000,3;800,5.30000000000000,3;1164.39873700000,5.39604800000000,3;1289.80436000000,5.42557900000000,3;1581.28769900000,5.60276200000000,3;1750.75475600000,5.80947600000000,3;2154.08635300000,5.80947600000000,3;2313.38538700000,5.57323200000000,3;2635.37279600000,5.35175200000000,3;2821.78655900000,5.35175200000000,3;3136.99528600000,5.05644600000000,3;3397.97455500000,4.30341700000000,3;3580.99897700000,3.50609100000000,3;3821.64219900000,1.74902100000000,3;3913.15441000000,1.39465400000000,3;3852.14626900000,0.936930000000000,3;3645.39645900000,0.789277000000000,3;3187.83540400000,0.582563000000000,3;2723.49566600000,0.553032000000000,3;2428.62298600000,0.553032000000000,3;2130.36096500000,0.553032000000000,3;1828.70960300000,0.553032000000000,3;1299.97238300000,0.582563000000000,3;1137.28400800000,0.641624000000000,3;998.321021000000,5.70611900000000,2.50000000000000;1415.20998200000,5.77994600000000,2.50000000000000;1988.00863700000,6.28196600000000,2.50000000000000;2337.11077500000,5.91283400000000,2.50000000000000;2469.29508000000,5.67658900000000,2.50000000000000;2984.47493500000,5.67658900000000,2.50000000000000;3259.01156800000,5.14503800000000,2.50000000000000;3540.32688300000,4.31818200000000,2.50000000000000;3831.81022200000,2.98930500000000,2.50000000000000;4001.27727900000,2.00003100000000,2.50000000000000;4119.90422000000,1.04028700000000,2.50000000000000;3994.49859700000,0.538267000000000,2.50000000000000;3557.27358900000,0.405379000000000,2.50000000000000;3059.04044000000,0.272491000000000,2.50000000000000;1462.66075800000,0.272491000000000,2.50000000000000;1299.97238300000,0.302022000000000,2.50000000000000;1001.71036200000,0.361083000000000,2.50000000000000;1191.51346600000,5.69135400000000,2.50000000000000;1588.06638100000,5.88330300000000,2.50000000000000;1791.42685000000,6.09001700000000,2.50000000000000;2167.64371700000,6.10478200000000,2.50000000000000;2628.59411400000,5.67658900000000,2.50000000000000;2794.67183000000,5.67658900000000,2.50000000000000;3713.18328200000,3.57991700000000,2.50000000000000;3770.80208100000,0.464440000000000,2.50000000000000;3292.90497900000,0.331553000000000,2.50000000000000;2692.99159600000,0.272491000000000,2.50000000000000;2143.91832900000,0.272491000000000,2.50000000000000;1767.70146200000,0.272491000000000,2.50000000000000;998.321021000000,6.04572100000000,2;1506.72219300000,6.04572100000000,2;1747.36541500000,6.31149600000000,2;1984.61929500000,6.54774100000000,2;2452.34837400000,6.04572100000000,2;2994.64295800000,6.04572100000000,2;3516.60149500000,4.98262000000000,2;4008.05596200000,3.03360100000000,2;4123.29356100000,1.88190800000000,2;4241.92050100000,1.08458300000000,2;4502.89977000000,0.0214810000000000,2;1174.56676100000,6.04572100000000,2;1333.86579500000,6.04572100000000,2;2208.31581100000,6.32626200000000,2;2594.70070200000,6.04572100000000,2;2801.45051200000,6.04572100000000,2;3174.27803900000,5.67658900000000,2;3333.57707300000,5.33698700000000,2;3638.61777600000,4.48060000000000,2;3821.64219900000,3.75710100000000,2;4065.67476100000,2.41345900000000,2;4374.10480600000,0.538267000000000,2;998.321021000000,6.28196600000000,1.50000000000000;1516.89021700000,6.29673100000000,1.50000000000000;1984.61929500000,6.75445500000000,1.50000000000000;2421.84430400000,6.54774100000000,1.50000000000000;2994.64295800000,6.54774100000000,1.50000000000000;3516.60149500000,5.44034400000000,1.50000000000000;4113.12553700000,3.04836700000000,1.50000000000000;4201.24840700000,2.02956100000000,1.50000000000000;4363.93678300000,1.06981700000000,1.50000000000000;4618.13736900000,0.0214810000000000,1.50000000000000;1157.62005500000,6.28196600000000,1.50000000000000;1344.03381800000,6.28196600000000,1.50000000000000;1754.14409700000,6.53297600000000,1.50000000000000;2171.03305900000,6.66586300000000,1.50000000000000;2598.09004300000,6.53297600000000,1.50000000000000;2808.22919500000,6.54774100000000,1.50000000000000;3174.27803900000,6.17860900000000,1.50000000000000;3347.13443800000,5.83900700000000,1.50000000000000;3703.01525800000,4.68731400000000,1.50000000000000;3909.76506800000,3.84569200000000,1.50000000000000;4489.34240500000,0.582563000000000,1.50000000000000;4285.98193600000,1.54230700000000,1.50000000000000;1001.71036200000,6.56250600000000,1;1516.89021700000,6.54774100000000,1;1981.22995400000,7.04976100000000,1;2994.64295800000,7.04976100000000,1;3523.38017700000,5.95712900000000,1;4248.69918300000,3.04836700000000,1;4302.92864200000,2.02956100000000,1;4550.35054600000,0.848338000000000,1;4753.71101500000,0.0214810000000000,1;800,6.50000000000000,1;1164.39873700000,6.53297600000000,1;1357.59118300000,6.53297600000000,1;1669.41056800000,6.72492500000000,1;1842.26696700000,6.88734300000000,1;2198.14778800000,7.03499600000000,1;2526.91387900000,7.03499600000000,1;2821.78655900000,7.03499600000000,1;3174.27803900000,6.68062900000000,1;3364.08114300000,6.31149600000000,1;3675.90052900000,5.33698700000000,1;3869.09297500000,4.56919200000000,1;4069.06410200000,3.78663100000000,1;4438.50228800000,1.37988800000000,1;4652.03078000000,0.434910000000000,1;998.321021000000,6.76922100000000,0.500000000000000;1499.94351100000,6.78398600000000,0.500000000000000;1750.75475600000,7.06452600000000,0.500000000000000;1981.22995400000,7.50748500000000,0.500000000000000;2998.03229900000,7.50748500000000,0.500000000000000;3757.24471700000,6.01619100000000,0.500000000000000;3862.31429200000,5.10074200000000,0.500000000000000;4001.27727900000,4.52489600000000,0.500000000000000;4248.69918300000,4.02287600000000,0.500000000000000;4374.10480600000,3.06313200000000,0.500000000000000;4391.05151200000,2.73829500000000,0.500000000000000;4407.99821700000,2.00003100000000,0.500000000000000;4506.28911100000,1.52754100000000,0.500000000000000;4760.48969700000,1.01075600000000,0.500000000000000;4885.89532000000,0.00671600000000000,0.500000000000000;1157.62005500000,6.75445500000000,0.500000000000000;1337.25513600000,6.75445500000000,0.500000000000000;2171.03305900000,7.49272000000000,0.500000000000000;2520.13519700000,7.49272000000000,0.500000000000000;2784.50380700000,7.50748500000000,0.500000000000000;3177.66738000000,7.16788300000000,0.500000000000000;3428.47862500000,6.66586300000000,0.500000000000000;4126.68290200000,4.27388600000000,0.500000000000000;4818.10849700000,0.523501000000000,0.500000000000000;4635.08407400000,1.29129700000000,0.500000000000000;1000,7.06452600000000,0;1482.99680500000,7.07929200000000,0;1927.00049600000,7.75849500000000,0;3011.58966400000,7.77326000000000,0;3804.69549300000,6.23767000000000,0;3977.55189100000,5.05644600000000,0;4302.92864200000,4.33294700000000,0;4523.23581700000,2.75306100000000,0;4560.51856900000,1.86714300000000,0;4821.49783800000,1.37988800000000,0;4967.23950700000,0.00671600000000000,0;6004.37789900000,0.00671600000000000,0;6007.76724000000,11.1988090000000,0;1000,11.1840430000000,0;800,7.20000000000000,0;800,9,0;800,11,0];

    map_data(:,3) = map_data(:,3) * abs(mod_CCP);

    engine_out = engine_tech_mod( engine_out, mod_CCP > 0, mod_CCP < 0, map_data );
end
%% Create strucutre of modifier information
    if exist('octane_knk_adjust_factor','var')
        engine_modifiers.octane_knk_adjust_factor = octane_knk_adjust_factor;
    else
        engine_modifiers.octane_knk_adjust_factor = nan;
    end
    if exist('octane_change','var')
        engine_modifiers.octane_change = octane_change;
        engine_modifiers.octane_modifier_used  = ~isempty(new_fuel) && fuel_octane_adjust &&  (octane_knk_adjust_factor~=0);
    else
        engine_modifiers.octane_change = 0;
         engine_modifiers.octane_modifier_used = 0;
    end

end


function [ engine] = engine_tech_mod( engine, AddTech, RmvTech, map_data )

    if ischar(map_data)
        load(map_data,'map_data');
    end

    %% scatteredInterpolant
    data_xscale = max(map_data(:,1)) - min(map_data(:,1));
    data_yscale = max(map_data(:,2)) - min(map_data(:,2));

    F = scatteredInterpolant( map_data(:,1)/data_xscale, map_data(:,2)/data_yscale, map_data(:,3), 'natural', 'linear' ); %extrapolant: nearest, linear, or none

    xin = (engine.fuel_map_speed_radps .* unit_convert.radps2rpm);    %Convert rad/s to RPM
    yin = (engine.fuel_map_torque_Nm) * 4 * pi / (engine.displacement_L /1000)*1e-5;     %Convert Nm to BMEP(2.4L)
    [x_grid_norm, y_grid_norm] =  meshgrid( xin/data_xscale, yin/data_yscale);

    bsfc_improve_pcnt = F( x_grid_norm, y_grid_norm );

    %% Update engine_new's fuel map
    if (AddTech)
        engine.fuel_map_gps = engine.fuel_map_gps .* (1-bsfc_improve_pcnt/100);
    elseif (RmvTech)
        engine.fuel_map_gps = engine.fuel_map_gps ./ (1-bsfc_improve_pcnt/100);
    end
    
    
    

end


function fuel_gps = min_fuel_ind_torque(engine, ind_torque_Nm)


    
    brk_torque_Nm = ind_torque_Nm + interp1( engine.closed_throttle_speed_radps, engine.closed_throttle_torque_Nm, engine.fuel_map_speed_radps,'linear','extrap');
    
    fuel_gps = interp2( engine.fuel_map_speed_radps, engine.fuel_map_torque_Nm, engine.fuel_map_gps, engine.fuel_map_speed_radps, brk_torque_Nm);

    fuel_gps = min(fuel_gps ./ engine.fuel_map_speed_radps);
    
end
