classdef P0_control < matlab.System
    % P0_control
    %
    % This template includes the minimum set of functions required
    % to define a System object with discrete state.
    
    % Public, tunable properties
    properties
        regenerative_braking_enable
        engine_nominal_idle_speed_radps
        P0_gear_ratio
        P0_gear_efficiency_norm
        ev_mode_min_soc_norm
        charge_depleting_min_soc_norm
        charge_sustaining_max_soc_norm
        emachine_motor_min_power_threshold_W
        charge_sustaining_min_power_W
            
    end
    
    properties(DiscreteState)
        engine_on_bool
        charge_sustain_state
        engine_off_cmd_time_sec
        prev_time_sec
    end
    
    % Pre-computed constants
    properties(Access = private)
        
    end
    
    methods(Access = protected)
        function setupImpl(obj)
            % Perform one-time calculations, such as computing constants

        end
        
        function [ engine_on_bool,...
                emachine_desired_torque_Nm, ...
                engine_desired_torque_Nm, ...
                engine_desired_power_W, ...
                driveline_desired_torque_Nm, ...
                driveline_desired_power_W, ...
                emachine_starting_torque_request_Nm, ...
                emachine_driveline_braking_torque_request_Nm, ...
                emachine_driveline_motoring_torque_request_Nm,...
                emachine_max_driveline_regen_power_W, ...
                emachine_max_driveline_regen_torque_Nm, ...
                emachine_max_driveline_motor_power_W, ...
                emachine_max_driveline_motor_torque_Nm, ...
                driveline_charge_sustaining_torque_request_Nm, ...
                torque_mode, ...
                charge_sustain_state] ...
                = ...
                stepImpl(obj, ...
                engine_on_start_stop, ...
                engine_off_allowed, ...
                driver_acl_norm, ...
                current_gear_num, ...
                vehicle_speed_mps, ...
                driver_accel_driveline_torque_request_Nm, ...
                driver_brake_driveline_torque_request_Nm, ...
                engine_speed_radps, ...
                battery_max_charge_power_W, ...
                battery_max_discharge_power_W, ...
                emachine_speed_radps, ...
                emachine_available_motor_torque_Nm, ...
                emachine_available_regen_torque_Nm, ...
                engine_max_power_W, ...
                engine_available_torque_Nm, ...
                engine_min_bsfc_torque_Nm, ...
                batt_SOC_norm, ...
                emachine_regen_derate_norm, ...
                emachine_motor_derate_norm, ...
                ev_mode_derate_norm, ...
                engine_inertia_torque_Nm, ...
                engine_tipin_torque_Nm, ...
                launch_device_lockup_bool)
            
            % Get delta time
            current_time_sec = getCurrentTime(obj);
            dT = current_time_sec - obj.prev_time_sec;
            obj.prev_time_sec = current_time_sec;
            
            CHARGE_DEPLETING  = 0;
            CHARGE_SUSTAINING = 1;
            
            ENGINE_ON  = true;
            ENGINE_OFF = false;
            
            
            engine_min_bsfc_power_W = engine_min_bsfc_torque_Nm * engine_speed_radps;
            
            emachine_available_regen_power_W = emachine_available_regen_torque_Nm * emachine_speed_radps;
            emachine_available_motor_power_W = emachine_available_motor_torque_Nm * emachine_speed_radps;
            
            emachine_max_driveline_regen_power_W = emachine_regen_derate_norm * min(battery_max_charge_power_W,    emachine_available_regen_power_W);
            emachine_max_driveline_motor_power_W = emachine_motor_derate_norm * min(battery_max_discharge_power_W, emachine_available_motor_power_W);
            
            emachine_max_driveline_regen_torque_Nm = min(obj.P0_gear_ratio * emachine_available_regen_torque_Nm, ...
                emachine_max_driveline_regen_power_W / max(1, engine_speed_radps)) / obj.P0_gear_efficiency_norm;
            
            emachine_max_driveline_motor_torque_Nm = min(obj.P0_gear_ratio * emachine_available_motor_torque_Nm, ...
                emachine_max_driveline_motor_power_W / max(1, engine_speed_radps)) * obj.P0_gear_efficiency_norm;
            
            driveline_desired_torque_Nm = driver_accel_driveline_torque_request_Nm - driver_brake_driveline_torque_request_Nm;
            driveline_desired_power_W   = driveline_desired_torque_Nm * engine_speed_radps;
            
            
            % charge sustaining state transitions
            if (batt_SOC_norm < obj.charge_depleting_min_soc_norm)
                obj.charge_sustain_state = CHARGE_SUSTAINING;
            end
            
            if (batt_SOC_norm > obj.charge_sustaining_max_soc_norm)
                obj.charge_sustain_state = CHARGE_DEPLETING;
            end
            
            % load-leveling torque calculation
            if (obj.engine_on_bool == ENGINE_ON) && ...
                    (obj.charge_sustain_state == CHARGE_SUSTAINING) && ...
                    (engine_speed_radps > obj.engine_nominal_idle_speed_radps * 2)
                if (driveline_desired_torque_Nm < engine_min_bsfc_torque_Nm)
                    driveline_charge_sustaining_torque_request_Nm = engine_min_bsfc_torque_Nm - driveline_desired_torque_Nm;
                else
                    driveline_charge_sustaining_torque_request_Nm = obj.charge_sustaining_min_power_W / engine_speed_radps;
                end
            else
                driveline_charge_sustaining_torque_request_Nm = 0;
            end
            
            % apply torque limit(s)
            driveline_charge_sustaining_torque_request_Nm = -min(emachine_max_driveline_regen_torque_Nm, ...
                driveline_charge_sustaining_torque_request_Nm);
            
            % engine starting / spin control
            if (engine_speed_radps < obj.engine_nominal_idle_speed_radps) && engine_on_start_stop && (current_gear_num < 2) % start engine / keep engine started
                emachine_starting_torque_request_Nm = interp1([0 1 2], [obj.P0_gear_ratio * emachine_available_motor_torque_Nm 0 0], engine_speed_radps / obj.engine_nominal_idle_speed_radps);
            else
                emachine_starting_torque_request_Nm = 0;
            end
            
            % torque control
            if (driveline_desired_torque_Nm < 0) || ~engine_on_start_stop % net negative torque requested (regen)
                % braking / regen or engine off
                torque_mode = -1;
                
                if ~engine_off_allowed
                    obj.engine_on_bool = ENGINE_ON;
                    obj.engine_off_cmd_time_sec=0;
                elseif (driver_brake_driveline_torque_request_Nm > 1) && (vehicle_speed_mps > 1)
                    obj.engine_off_cmd_time_sec = obj.engine_off_cmd_time_sec + dT;
                    if  obj.engine_off_cmd_time_sec > 5  % wait 5 seconds before cutting fuel
                        obj.engine_on_bool = ENGINE_OFF;
                    else
                        obj.engine_on_bool = ENGINE_ON;
                    end
                    
                elseif (vehicle_speed_mps < 0.5)
                    obj.engine_on_bool = engine_on_start_stop;
                    obj.engine_off_cmd_time_sec = (obj.engine_off_cmd_time_sec + dT)*(engine_on_start_stop==0);
                else
                    obj.engine_on_bool = obj.engine_on_bool && engine_on_start_stop;
                    obj.engine_off_cmd_time_sec = (obj.engine_off_cmd_time_sec + dT)*(engine_on_start_stop==0);
                end
                
                emachine_driveline_motoring_torque_request_Nm = 0;
                
                if obj.regenerative_braking_enable
                    emachine_driveline_braking_torque_request_Nm = interp1([-Inf obj.engine_nominal_idle_speed_radps 2*obj.engine_nominal_idle_speed_radps Inf], ...
                        [0 0 driver_brake_driveline_torque_request_Nm driver_brake_driveline_torque_request_Nm], engine_speed_radps, 'linear');
                else
                    emachine_driveline_braking_torque_request_Nm = 0;
                end
                
                % apply torque limit(s)
                emachine_driveline_braking_torque_request_Nm = -min(emachine_max_driveline_regen_torque_Nm, ...
                    emachine_driveline_braking_torque_request_Nm);
                
                engine_desired_torque_Nm = driveline_desired_torque_Nm - emachine_driveline_braking_torque_request_Nm;
                
            else % torque positive and/or engine on request
                emachine_driveline_braking_torque_request_Nm = 0;
                
                if (obj.charge_sustain_state == CHARGE_DEPLETING) && ...
                        (driveline_desired_torque_Nm > engine_min_bsfc_torque_Nm) && ...
                        (emachine_max_driveline_motor_torque_Nm > 0) && ...
                        ((driveline_desired_power_W - engine_min_bsfc_power_W) > obj.emachine_motor_min_power_threshold_W) && ...
                        (emachine_max_driveline_motor_power_W > obj.emachine_motor_min_power_threshold_W)
                    % load reduce to engine min bsfc mode
                    obj.engine_on_bool = ENGINE_ON;
                    emachine_driveline_motoring_torque_request_Nm = driveline_desired_torque_Nm - engine_min_bsfc_torque_Nm;   % load reduce above min BSFC curve
                    torque_mode = 1;
                elseif (obj.charge_sustain_state == CHARGE_DEPLETING) && (driveline_desired_torque_Nm > engine_available_torque_Nm)
                    % torque assist mode
                    obj.engine_on_bool = ENGINE_ON;
                    torque_mode = 2;
                    emachine_driveline_motoring_torque_request_Nm = driveline_desired_torque_Nm - engine_available_torque_Nm;
                elseif (obj.charge_sustain_state == CHARGE_DEPLETING) && ...
                        (batt_SOC_norm >= obj.ev_mode_min_soc_norm) && ...
                        (driveline_desired_power_W < emachine_max_driveline_motor_power_W)
                    % electric-drive mode
                    torque_mode = 3;
                    emachine_driveline_motoring_torque_request_Nm = driveline_desired_torque_Nm * ev_mode_derate_norm;
                else
                    % passive mode
                    obj.engine_on_bool = (driveline_desired_power_W > emachine_max_driveline_motor_power_W) || ...
                        (obj.engine_on_bool == ENGINE_ON) || ~engine_off_allowed;
                    emachine_driveline_motoring_torque_request_Nm = 0;
                    torque_mode = 0;
                end
                
                % apply torque limit(s)
                emachine_driveline_motoring_torque_request_Nm = min(emachine_max_driveline_motor_torque_Nm, ...
                    emachine_driveline_motoring_torque_request_Nm);
                
                engine_desired_torque_Nm = max(0, driveline_desired_torque_Nm - ...
                    emachine_driveline_motoring_torque_request_Nm - ...
                    driveline_charge_sustaining_torque_request_Nm);
                
            end % torque positive and/or engine on request
            
            emachine_driveline_braking_torque_request_Nm  = emachine_driveline_braking_torque_request_Nm + engine_inertia_torque_Nm;
            emachine_driveline_motoring_torque_request_Nm = emachine_driveline_motoring_torque_request_Nm + engine_tipin_torque_Nm;
            
            % calculate net MG torque request
            emachine_desired_torque_Nm = emachine_starting_torque_request_Nm + driveline_charge_sustaining_torque_request_Nm + emachine_driveline_braking_torque_request_Nm  + emachine_driveline_motoring_torque_request_Nm;
            
            
            
            % compensate for starter-generator gear ratio and gear efficiency
            if emachine_desired_torque_Nm > 0
                emachine_desired_torque_Nm = emachine_desired_torque_Nm / obj.P0_gear_ratio / obj.P0_gear_efficiency_norm;
            else
                emachine_desired_torque_Nm = emachine_desired_torque_Nm / obj.P0_gear_ratio * obj.P0_gear_efficiency_norm;
            end
            
            if obj.engine_on_bool == ENGINE_OFF
                engine_desired_torque_Nm = 0;
            end
            
            if driver_acl_norm < 0.99
                engine_desired_power_W = engine_desired_torque_Nm * engine_speed_radps;
            else
                engine_desired_power_W = engine_max_power_W * 2; % punch it!
            end
            
            % output variables based off persistent variables
            engine_on_bool        = obj.engine_on_bool;
            charge_sustain_state  = obj.charge_sustain_state;
            
            
            
        end
        
        function resetImpl(obj)
            % Initialize / reset discrete-state properties
            obj.engine_on_bool          = false;
            obj.charge_sustain_state    = 0;
            obj.engine_off_cmd_time_sec = 0;
            obj.prev_time_sec = getCurrentTime(obj);
        end
    end
end
