classdef P2_control < matlab.System
    % P2 engine/drive motor controls
    %
    % This template includes the minimum set of functions required
    % to define a System object with discrete state.

    % Public, tunable properties
    properties

    end

    properties(DiscreteState)
        engine_on_off_state
        charge_sustain_state
        charge_sustain_mode
        state_time
        torque_mode
        desired_torque_mode
        engine_commanded_torque_Nm
        engine_on_time_secs
        regen_state
        time
    end

    % Pre-computed constants
    properties(Access = private)
        TORQUE_MODE_EV     = 0;
        TORQUE_MODE_HYBRID = 1;
        TORQUE_MODE_HYBRID_TO_EV = 2;

        ENGINE_OFF = 0;
        ENGINE_ON  = 1;

        CHARGE_DEPLETING  = 0;
        CHARGE_SUSTAINING = 1;

        CHARGE_SUSTAINING_DOWN = 0;
        CHARGE_SUSTAINING_UP   = 1;

        REGEN_DISABLED = 0;
        REGEN_ENABLED = 1;
    end

    methods(Access = protected)
        function setupImpl(obj)
            % Perform one-time calculations, such as computing constants
        end

        function ...
                [drive_motor_torque_request_Nm, ...
                engine_desired_power_W, ...
                engine_commanded_torque_Nm, ...
                powerpack_torque_request_Nm,  ...
                engine_on_off_state, ...
                ...
                powerpack_power_request_kW, ...
                battery_limited_emachine_max_regen_torque_Nm, ...
                battery_limited_emachine_max_motor_torque_Nm, ...
                emachine_regen_torque_limit_Nm, ...
                emachine_motor_torque_limit_Nm, ...
                charge_sustain_state, ...
                charge_sustain_mode, ...
                desired_torque_mode, ...
                torque_mode, ...
                engine_idle_mode, ...
                engine_on_time_secs, ...
                engine_on_off_reason] ...
                = stepImpl(...
                obj, ...
                engine_off_allowed_bool, ...
                driver_wheeltorque_request_Nm, ...
                driver_wheelpower_request_W, ...
                driver_accel_power_request_W, ...
                driver_accel_driveline_torque_request_Nm, ...
                driver_brake_driveline_torque_request_Nm, ...
                driver_max_wheeltorque_Nm, ...
                vehicle_speed_mps, ...
                engine_speed_radps, ...
                engine_min_torque_Nm, ...
                P2_speed_radps, ...
                P2_motor_max_torque_Nm, ...
                P2_regen_max_torque_Nm, ...
                battery_soc_norm, ...
                battery_max_charge_power_W, ...
                battery_max_discharge_power_W, ...
                engine_min_bsfc_torque_Nm, ...
                motor_soc_derate_norm, ...
                regen_soc_derate_norm, ...
                regen_vspeed_derate_norm, ...
                P2_clutch_state, ...
                engine_actual_torque_Nm, ...
                P2_max_power_W, ...
                P2_gear_ratio, ...
                battery_min_soc_norm, ...
                battery_hyst_soc_norm, ...
                battery_loadlevel_soc_norm, ...
                engine_min_on_time_secs, ...
                engine_min_off_time_secs, ...    
                engine_off_delay_secs, ...
                max_EV_vehicle_speed_mps, ...
                engine_torque_rate_factor, ...
                dt ...
                ) ...                
            %% Dynamic Inits
            if obj.time == 0     
                if (battery_soc_norm > battery_loadlevel_soc_norm)
                    obj.charge_sustain_state = obj.CHARGE_DEPLETING;
                else
                    obj.charge_sustain_state = obj.CHARGE_SUSTAINING;
                end
            end
            
            engine_on_off_reason = 0;

            P2_clutch_engaged = (P2_clutch_state == 1);
            
            if P2_clutch_engaged
                engine_torque_Nm = engine_actual_torque_Nm;
            else
                engine_torque_Nm = 0;
            end

            %% Unit Conversions
            driver_wheelpower_request_kW = driver_wheelpower_request_W / 1000;
            driver_accel_power_request_kW = driver_accel_power_request_W / 1000;
            emachine_max_power_kW = P2_max_power_W / 1000;
            
            %% Motor Limits

            battery_limited_emachine_max_regen_torque_Nm = battery_max_charge_power_W / max(0.1, P2_speed_radps);
            battery_limited_emachine_max_motor_torque_Nm = battery_max_discharge_power_W / max(0.1, P2_speed_radps);

            emachine_motor_torque_limit_Nm = motor_soc_derate_norm * min(battery_limited_emachine_max_motor_torque_Nm, P2_motor_max_torque_Nm);

            %% mode determination of regen enable
            if obj.regen_state == obj.REGEN_DISABLED
                emachine_regen_torque_limit_Nm = 0;

                if regen_vspeed_derate_norm == 1
                    obj.regen_state = obj.REGEN_ENABLED;
                end
            else % regen_state == REGEN_ENABLED
                emachine_regen_torque_limit_Nm = min(regen_vspeed_derate_norm, regen_soc_derate_norm) * min(battery_limited_emachine_max_regen_torque_Nm, P2_regen_max_torque_Nm);

                if regen_vspeed_derate_norm == 0
                    obj.regen_state = obj.REGEN_DISABLED;
                end
            end
            
            emachine_generator_torque_limit_Nm = regen_soc_derate_norm * min(battery_limited_emachine_max_regen_torque_Nm, P2_regen_max_torque_Nm);
                        
            powerpack_torque_request_Nm = driver_accel_driveline_torque_request_Nm - min(emachine_regen_torque_limit_Nm, driver_brake_driveline_torque_request_Nm);
            powerpack_power_request_kW = powerpack_torque_request_Nm * P2_speed_radps / P2_gear_ratio / 1000;

%             % Motor Saturation Flag, not currently used
%             if driver_accel_driveline_torque_request_Nm > 0.73 * emachine_motor_torque_limit_Nm
%                 obj.emachine_saturated = true;
%                 obj.emachine_sat_time_secs = 0;
%             elseif driver_accel_driveline_torque_request_Nm < 0.5 * emachine_motor_torque_limit_Nm
%                 obj.emachine_sat_time_secs = obj.emachine_sat_time_secs + dt;
%                 if obj.emachine_sat_time_secs > 3 % 3 SECS.
%                     obj.emachine_saturated = false;
%                 end
%             end

            %% mode determination of charge-depleting / sustaining
            if obj.charge_sustain_state == obj.CHARGE_DEPLETING
                if battery_soc_norm <= battery_min_soc_norm
                    obj.charge_sustain_state = obj.CHARGE_SUSTAINING;
                end

                % engine on/off logic, charge depleting
                if (obj.torque_mode == obj.TORQUE_MODE_EV) && ...
                        ((vehicle_speed_mps > max_EV_vehicle_speed_mps) || ... 
                         (driver_accel_power_request_kW >= 1.05 * emachine_max_power_kW))

                    obj.desired_torque_mode = obj.TORQUE_MODE_HYBRID;
                    engine_on_off_reason = 5;
                    
                end 

                if (obj.torque_mode == obj.TORQUE_MODE_HYBRID) && ...
                        ((driver_accel_power_request_kW <= 0.5 * emachine_max_power_kW))

                    obj.desired_torque_mode = obj.TORQUE_MODE_EV;
                    engine_on_off_reason = -3;
                end 

            else % charge_sustain_state == CHARGE_SUSTAINING

                % CHARGE SUSTAIN MODE DETERMINATION
                if obj.charge_sustain_mode == obj.CHARGE_SUSTAINING_UP && (battery_soc_norm >= battery_loadlevel_soc_norm - battery_hyst_soc_norm)
                    obj.charge_sustain_mode = obj.CHARGE_SUSTAINING_DOWN;
                end

                if obj.charge_sustain_mode == obj.CHARGE_SUSTAINING_DOWN && (battery_soc_norm <= battery_min_soc_norm - battery_hyst_soc_norm)
                    obj.charge_sustain_mode = obj.CHARGE_SUSTAINING_UP;
                end

                % launch start
                if obj.torque_mode == obj.TORQUE_MODE_EV && ...
                    ((vehicle_speed_mps > 1) && (vehicle_speed_mps <= 2) && ...
                        ((driver_wheeltorque_request_Nm > 0) && (driver_brake_driveline_torque_request_Nm == 0)) && ...
                        (battery_soc_norm <= battery_loadlevel_soc_norm - battery_hyst_soc_norm))

                    obj.desired_torque_mode = obj.TORQUE_MODE_HYBRID;

                    engine_on_off_reason = 1;
                end

                % high torque start
                if obj.torque_mode == obj.TORQUE_MODE_EV && ...
                    ((driver_wheeltorque_request_Nm >= 0.106 * driver_max_wheeltorque_Nm) && (vehicle_speed_mps >= 0.1) && ...
                        (battery_soc_norm < battery_loadlevel_soc_norm - battery_hyst_soc_norm))

                    obj.desired_torque_mode = obj.TORQUE_MODE_HYBRID;

                    engine_on_off_reason = 2;
                end

                if vehicle_speed_mps >= 27.5
                        turnon_power_kW = emachine_max_power_kW;
                elseif vehicle_speed_mps >= 17.5
                        turnon_power_kW = emachine_max_power_kW / 2;
                    else
                        turnon_power_kW = emachine_max_power_kW / 2;
                end

                % power start
                if obj.torque_mode == obj.TORQUE_MODE_EV && ...
                        (vehicle_speed_mps > 2) && ...
                        ((driver_wheelpower_request_kW >= turnon_power_kW) && ... 
                        (battery_soc_norm < battery_loadlevel_soc_norm))

                    obj.desired_torque_mode = obj.TORQUE_MODE_HYBRID;

                    engine_on_off_reason = 3;
                end

                % SoC start *** 
                if obj.torque_mode == obj.TORQUE_MODE_EV && ...                    
                        (battery_soc_norm <= battery_min_soc_norm)

                    obj.desired_torque_mode = obj.TORQUE_MODE_HYBRID;

                    engine_on_off_reason = 4;
                end
                                
                if obj.charge_sustain_mode == obj.CHARGE_SUSTAINING_UP
                        if vehicle_speed_mps >= 17.5
                            shutoff_power_kW = -0.2 * emachine_max_power_kW;
                        else
                            shutoff_power_kW = -0.15 * emachine_max_power_kW;
                        end
                    else
                        if vehicle_speed_mps >= 17.5
                            shutoff_power_kW = 0.1 * emachine_max_power_kW;
                        else
                            shutoff_power_kW = 0;
                        end
                end

                if obj.torque_mode == obj.TORQUE_MODE_HYBRID && ...
                        (vehicle_speed_mps > 2) && ...
                        (((driver_wheelpower_request_kW < shutoff_power_kW) && ((battery_soc_norm > battery_min_soc_norm + battery_hyst_soc_norm))) || ... 
                        (battery_soc_norm > battery_loadlevel_soc_norm))

                    obj.desired_torque_mode = obj.TORQUE_MODE_EV;

                    engine_on_off_reason = -1;
                end

                if obj.torque_mode == obj.TORQUE_MODE_HYBRID && ...
                        (vehicle_speed_mps >= 29) && ...
                        (((driver_wheelpower_request_kW < 0.4 * emachine_max_power_kW) && ((battery_soc_norm > battery_min_soc_norm + battery_hyst_soc_norm))) || ... 
                        (battery_soc_norm > battery_loadlevel_soc_norm))

                    obj.desired_torque_mode = obj.TORQUE_MODE_EV;

                    engine_on_off_reason = -2;
                end

            end

            %% mode determination with engine_off_allowed_bool
            if engine_off_allowed_bool == false

                obj.desired_torque_mode = obj.TORQUE_MODE_HYBRID;

                engine_idle_mode = (battery_soc_norm > battery_min_soc_norm);

            else
                
                engine_idle_mode = false;
            
            end

            %% EV and HEV Mode Operation Method

            % torque_mode = TORQUE_MODE_EV;                                    
            if obj.torque_mode == obj.TORQUE_MODE_EV  % pure EV mode    
                obj.state_time = obj.state_time + dt;
                
                if engine_torque_Nm <= 0
                    obj.engine_on_off_state = obj.ENGINE_OFF;
                end
                
                engine_desired_torque_Nm = engine_min_torque_Nm; % need to request a negative torque for engine speed to drop when engine is off
                
                if (obj.desired_torque_mode == obj.TORQUE_MODE_HYBRID) && (obj.state_time > engine_min_off_time_secs)
                    obj.torque_mode = obj.desired_torque_mode;
                    obj.state_time = 0;
                end

            else % if obj.torque_mode == obj.TORQUE_MODE_HYBRID
                obj.state_time = obj.state_time + dt;

                obj.engine_on_off_state = obj.ENGINE_ON;
                                
                if engine_idle_mode == false

                    if P2_clutch_engaged
                        if powerpack_torque_request_Nm > 0 && obj.desired_torque_mode == obj.TORQUE_MODE_HYBRID
                            
                            engine_desired_torque_Nm = engine_min_bsfc_torque_Nm;
                            
                        else  % powerpack_torque_request_Nm < 0 or Regen Condition
                        
                            engine_desired_torque_Nm = min(engine_min_bsfc_torque_Nm, min(0, powerpack_torque_request_Nm) + emachine_generator_torque_limit_Nm);
                        
                        end
                    else % during engine startup, command load to assist startup and ramp-in
                            engine_desired_torque_Nm = engine_min_bsfc_torque_Nm; % was -engine_min_torque_Nm with prior engine model
                    end
                        
                else % When engine_idle_mode = true % Cold start and run condition
                    obj.state_time = obj.state_time + dt;
                    
                    engine_desired_torque_Nm = 0;      
                end
                
                if obj.desired_torque_mode == obj.TORQUE_MODE_EV && (obj.state_time >= engine_min_on_time_secs)
                    obj.torque_mode = obj.TORQUE_MODE_EV;
                    obj.state_time = 0;
                end                
            end
            
            obj.engine_commanded_torque_Nm = obj.engine_commanded_torque_Nm + (engine_desired_torque_Nm - obj.engine_commanded_torque_Nm) * engine_torque_rate_factor;
                           
            drive_motor_torque_request_Nm = max(-emachine_generator_torque_limit_Nm, min(emachine_motor_torque_limit_Nm, powerpack_torque_request_Nm - engine_torque_Nm));
            
            if vehicle_speed_mps <= 0 && powerpack_torque_request_Nm <= 0
                drive_motor_torque_request_Nm = 0;
            end
            
            %%
            if obj.engine_on_off_state == obj.ENGINE_ON
                obj.engine_on_time_secs = obj.engine_on_time_secs + dt;
            else % ENGINE_OFF
                obj.engine_on_time_secs = 0;
%                 engine_desired_torque_Nm = engine_min_torque_Nm;    % need to request a negative torque for engine speed to drop when engine is off
            end
                        
            % output variables
            engine_desired_power_W     = engine_desired_torque_Nm * engine_speed_radps;
            engine_on_off_state        = obj.engine_on_off_state;
            charge_sustain_state       = obj.charge_sustain_state;
            charge_sustain_mode        = obj.charge_sustain_mode;
            desired_torque_mode        = obj.desired_torque_mode;
            torque_mode                = obj.torque_mode;
            engine_on_time_secs        = obj.engine_on_time_secs;
            engine_commanded_torque_Nm = obj.engine_commanded_torque_Nm;
            
            obj.time = obj.time + dt;
        end

        function resetImpl(obj)
            % Initialize / reset discrete-state properties
            obj.time = 0;
    
            obj.charge_sustain_state = obj.CHARGE_DEPLETING;

            obj.charge_sustain_mode  = obj.CHARGE_SUSTAINING_DOWN;
        %   obj.charge_sustain_mode  = obj.CHARGE_SUSTAINING_UP;

            obj.engine_on_off_state = obj.ENGINE_OFF;
            
            obj.state_time = 0;
            
            obj.engine_commanded_torque_Nm = 0;
            
            obj.torque_mode = obj.TORQUE_MODE_EV;
            obj.desired_torque_mode = obj.TORQUE_MODE_EV;

            obj.engine_on_time_secs = 0;

            obj.regen_state = obj.REGEN_DISABLED;
        end
        
    end
end
