function REVS_publish_emachine_minipacket( gen_file , date)
%REVS_PUBLISH_EMACHINE_MINIPACKET
%	Run complete process to create documentation for an engine input to REVS and store into zip file
%		- Publish parameter generation
%		- Export PDF versions of Efficiency Map and Power loss map
%		- Export CSV table of resulting electrical power consumption map


full_file = which(gen_file);

if isempty( full_file )
	error('Unable to locate "%s"', gen_file );
end

[file_path, file_name, file_ext] = fileparts( full_file);


cd( file_path );


doc_file = REVS_publish_param_gen( gen_file);


if nargin < 2
	date = now();
end


% Let files close out so when they are opened below they 
pause(0.1);

out_m_file = [file_path,'\',regexprep( file_name, '^gen_',''),'.m'];
out_zip_file = [file_path,'\',regexprep( file_name, '^gen_',''),'.zip'];

% Load Created emachine
run( out_m_file )


citation_tex = sprintf('\\it%s - ALPHA Map Package.\\rm Version %s.\nAnn Arbor MI: US EPA National Vehicle and Fuel Emissions Laboratory, National Center for Advanced Technology, %s.', mg.name, datestr(date,'yyyy-mm'), datestr(date,'yyyy'));
citation_html = sprintf('<I>%s - ALPHA Map Package.</I> Version %s. Ann Arbor MI: US EPA National Vehicle and Fuel Emissions Laboratory, National Center for Advanced Technology, %s.', mg.name, datestr(date,'yyyy-mm'), datestr(date,'yyyy'));
citation_raw = sprintf('%s - ALPHA Map Package. Version %s. Ann Arbor MI: US EPA National Vehicle and Fuel Emissions Laboratory, National Center for Advanced Technology, %s.', mg.name, datestr(date,'yyyy-mm'), datestr(date,'yyyy'));


% Insert Citation into process dod HTML
fid = fopen( doc_file, 'r+');
html_text = fread(fid,'*char')';
insert_str = sprintf('<p style="clear:both">SUGGESTED CITATION:<br>%s</p><!--/introduction-->', citation_html);
html_text = strrep( html_text, '<!--/introduction-->', insert_str);
fseek(fid,0,-1);
fprintf(fid,'%s',html_text);
fclose(fid);



% Insert Citation into header of m file
fid = fopen(  out_m_file ,'r+');
m_file_text = textscan(fid,'%s','delimiter','\n','Whitespace','');
m_file_text = m_file_text{1};
m_file_text(7:end+4) = m_file_text(3:end);
m_file_text{3} = '';
m_file_text{4} = '% SUGGESTED CITATION:';
m_file_text{5} = ['% ',citation_raw];
m_file_text{6} = '';
fseek(fid,0,-1);
fprintf(fid,'%s\n', m_file_text{:});
fclose(fid);


mkdir('minipacket_temp');

copyfile(doc_file,['minipacket_temp/3- ', mg.name,' - ALPHA Generation Process.html'])
copyfile(out_m_file,['minipacket_temp/7- ',regexprep( file_name, '^gen_',''),'.m'])


% Efficiency
REVS_plot_emachine(mg,'efficiency', 'title',mg.name,'string','show_point_labels',true);
set(gca,'Position',[ 0.13, 0.17, 0.775, 0.74]);
annotation('textbox',[0.015,0.0,0.98,0.09],'string',['SUGGESTED CITATION:',10,citation_tex],'FontSize',8,'LineStyle','none','VerticalAlignment','bottom','FitBoxtoText','off');
export_pdf( ['minipacket_temp/4a- ', mg.name,' - Efficiency.pdf'])
export_emf( ['minipacket_temp/4b- ', mg.name,' - Efficiency.emf'])

% Power loss
REVS_plot_emachine(mg,'power loss', 'title',mg.name,'string','show_point_labels',true);
set(gca,'Position',[ 0.13, 0.17, 0.775, 0.74]);
annotation('textbox',[0.015,0.0,0.98,0.09],'string',['SUGGESTED CITATION:',10,citation_tex],'FontSize',8,'LineStyle','none','VerticalAlignment','bottom','FitBoxtoText','off');
export_pdf( ['minipacket_temp/5a- ', mg.name,' - Power loss.pdf'])
export_emf( ['minipacket_temp/5b- ', mg.name,' - Power loss.emf'])



% spreadsheet containing the maps
export_fuelmap_spreadsheet( mg , ['minipacket_temp/6- ', mg.name,' - Electrical Power Consumption Data.xlsx'], citation_raw )

cd minipacket_temp
zip( out_zip_file,'*.*')
cd ..

rmdir('minipacket_temp','s')

end

function export_pdf( file )

fig = gcf;

set(fig,'PaperSize', [8.5,11], 'PaperOrientation','landscape','PaperPosition', [0 0 11 8.5]);

style = hgexport('factorystyle');
style.Format = 'pdf';
style.PSLevel = 2;
style.Width = 11.0;
style.Height = 8.5;
style.ScaledFontSize = 150;

hgexport(fig,file,style)

end

function export_emf( file )

fig = gcf;

set(fig,'color','none','Inverthardcopy','off');

style = hgexport('factorystyle');

style.Format = 'meta';
style.Width = 6.5  ;
style.Height = 4.9;
style.ScaledFontSize = 100;

hgexport(fig,file,style)

end

function export_fuelmap_spreadsheet( mg , file, citation)

fuel_map_speed_radps = mg.electric_power_W.get_signal_breakpoints( 'emach_spd_radps');
fuel_map_torque_Nm = mg.electric_power_W.get_signal_breakpoints( 'emach_trq_Nm'); 

positive_full_throttle_speed_radps = mg.positive_torque_limit_Nm.axis_1.breakpoints;
positive_full_throttle_torque_Nm = mg.positive_torque_limit_Nm.table  ;
negative_full_throttle_speed_radps = mg.negative_torque_limit_Nm.axis_1.breakpoints;
negative_full_throttle_torque_Nm = mg.negative_torque_limit_Nm.table  ;

[speed_radps, torque_Nm] = meshgrid(fuel_map_speed_radps, fuel_map_torque_Nm);
max_torque_Nm = interp1(positive_full_throttle_speed_radps, positive_full_throttle_torque_Nm, speed_radps );
min_torque_Nm = interp1(negative_full_throttle_speed_radps, negative_full_throttle_torque_Nm, speed_radps );

electric_power_kW = mg.electric_power_W.table(:)/1000;
efficiency_map = mg.efficiency_map_norm.table*100;

if isempty( mg.electric_power_W.axis_3.breakpoints )
    voltage_V = ones(size(efficiency_map)) * mg.positive_torque_limit_Nm.axis_2.breakpoints;
else
    voltage_V = [];
    speed_radps_dim = [];
    torque_Nm_dim = [];
    
    for i = 1:size(mg.efficiency_map_norm.table,3)
        voltage_V_dim = ones(size(mg.efficiency_map_norm.table,1).*size(mg.efficiency_map_norm.table,2),1) .*  round(mg.efficiency_map_norm.axis_3.breakpoints(i));
        voltage_V = [voltage_V;voltage_V_dim];
        
        speed_radps_dim = [speed_radps_dim; speed_radps(:)];
        torque_Nm_dim = [torque_Nm_dim; torque_Nm(:)];
    end
    speed_radps = speed_radps_dim;
    torque_Nm = torque_Nm_dim;
end

keep = torque_Nm(:) <= max_torque_Nm(:) & torque_Nm(:) >= min_torque_Nm(:);

sheet = 'emachine Map';

xls = xlseditor( file);

xls.add_sheet(sheet);
xls.delete_sheet('Sheet1');

xls.write({[mg.name,' - emachine Map Data'];'SUGGESTED CITATION:';citation;''},'A1:A3');


	header =	{	'Speed',	'Torque',	'Electrical Power Consumption',	'Efficiency',	'Voltage';
					'RPM',		'Nm',		'kW',		                    '%',	        'V'		   };
	
	xls.write(header, 'A5')
	
	data = [ speed_radps(:)* unit_convert.radps2rpm, torque_Nm(:), electric_power_kW(:), efficiency_map(:), voltage_V(:)];
	data(~keep,:) = [];
	
	xls.write( data,'A7')
	xls.format( 'A:A','NumberFormat','0','ColWidth',19);
	xls.format( 'B:B','NumberFormat','0.00','ColWidth',19);	
    xls.format( 'C:C','NumberFormat','0.00','ColWidth',19);
	xls.format( 'D:D','NumberFormat','0.0','ColWidth',19);
    xls.format( 'E:E','NumberFormat','0','ColWidth',19);

	
	xls.format('A1:E1','MergeCells',true)
	xls.format('A2:E2','MergeCells',true)
	xls.format('A3:E3','MergeCells',true)
	
	
	xls.format('A1','HorizAlign','center','FontBold',true);	
	xls.format('A5:H6','HorizAlign','center','FontBold',true);
	
	xls.format('3:3','WrapText',true,'RowHeight',30);
	

xls.close(1);

end



