function REVS_publish_engine_minipacket( gen_file , date)
%REVS_PUBLISH_ENGINE_MINIPACKET
%	Run complete process to create documentation for an engine input to REVS and store into zip file
%		- Publish parameter generation
%		- Export PDF versions of BSFC and Efficiency Map
%		- Export CSV table of resulting fuel map


full_file = which(gen_file);

if isempty( full_file )
	error('Unable to locate "%s"', gen_file );
end

[file_path, file_name, file_ext] = fileparts( full_file);


cd( file_path );


doc_file = REVS_publish_param_gen( gen_file);


if nargin < 2
	date = now();
end


% Let files close out so when they are opened below they 
pause(0.1);

out_m_file = [file_path,'\',regexprep( file_name, '^gen_',''),'.m'];
out_zip_file = [file_path,'\',regexprep( file_name, '^gen_',''),'.zip'];

% Load Created Engine
run( out_m_file )




citation_tex = sprintf('\\it%s - ALPHA Map Package.\\rm Version %s.\nAnn Arbor MI: US EPA National Vehicle and Fuel Emissions Laboratory, National Center for Advanced Technology, %s.', engine.name, datestr(date,'yyyy-mm'), datestr(date,'yyyy'));
citation_html = sprintf('<I>%s - ALPHA Map Package.</I> Version %s. Ann Arbor MI: US EPA National Vehicle and Fuel Emissions Laboratory, National Center for Advanced Technology, %s.', engine.name, datestr(date,'yyyy-mm'), datestr(date,'yyyy'));
citation_raw = sprintf('%s - ALPHA Map Package. Version %s. Ann Arbor MI: US EPA National Vehicle and Fuel Emissions Laboratory, National Center for Advanced Technology, %s.', engine.name, datestr(date,'yyyy-mm'), datestr(date,'yyyy'));


% Insert Citation into process dod HTML
fid = fopen( doc_file, 'r+');
html_text = fread(fid,'*char')';
insert_str = sprintf('<p style="clear:both">SUGGESTED CITATION:<br>%s</p><!--/introduction-->', citation_html);
html_text = strrep( html_text, '<!--/introduction-->', insert_str);
fseek(fid,0,-1);
fprintf(fid,'%s',html_text);
fclose(fid);



% Insert Citation into header of m file
fid = fopen(  out_m_file ,'r+');
m_file_text = textscan(fid,'%s','delimiter','\n','Whitespace','');
m_file_text = m_file_text{1};
m_file_text(7:end+4) = m_file_text(3:end);
m_file_text{3} = '';
m_file_text{4} = '% SUGGESTED CITATION:';
m_file_text{5} = ['% ',citation_raw];
m_file_text{6} = '';
fseek(fid,0,-1);
fprintf(fid,'%s\n', m_file_text{:});
fclose(fid);






mkdir('minipacket_temp');

copyfile(doc_file,['minipacket_temp/3- ', engine.name,' - ALPHA Generation Process.html'])
copyfile(out_m_file,['minipacket_temp/7- ',regexprep( file_name, '^gen_',''),'.m'])


if engine.has_deac
	
% BSFC Plot
REVS_plot_engine( engine, 'BSFC', 'show_bmep','no_min_bsfc','no_na_curve', 'show_redline','deac_prop','Enabled' );
set(gca,'Position',[ 0.13, 0.17, 0.775, 0.74]);
annotation('textbox',[0.015,0.0,0.98,0.09],'string',['SUGGESTED CITATION:',10,citation_tex],'FontSize',8,'LineStyle','none','VerticalAlignment','bottom','FitBoxtoText','off');
export_pdf( ['minipacket_temp/4a- ', engine.name,' - Deac Enabled - BSFC.pdf'])
export_emf( ['minipacket_temp/4b- ', engine.name,' - Deac Enabled - BSFC.emf'])

% BTE Plot
REVS_plot_engine( engine, 'efficiency', 'show_bmep','no_min_bsfc','no_na_curve' , 'show_redline','deac_prop','Enabled');
set(gca,'Position',[ 0.13, 0.17, 0.775, 0.74]);
annotation('textbox',[0.015,0.0,0.98,0.09],'string',['SUGGESTED CITATION:',10,citation_tex],'FontSize',8,'LineStyle','none','VerticalAlignment','bottom','FitBoxtoText','off');
export_pdf( ['minipacket_temp/5a- ', engine.name,' - Deac Enabled - Efficiency.pdf'])
export_emf( ['minipacket_temp/5b- ', engine.name,' - Deac Enabled - Efficiency.emf'])
	
% BSFC Plot
REVS_plot_engine( engine, 'BSFC', 'show_bmep','no_min_bsfc','no_na_curve', 'show_redline','deac_prop','Disabled' );
set(gca,'Position',[ 0.13, 0.17, 0.775, 0.74]);
annotation('textbox',[0.015,0.0,0.98,0.09],'string',['SUGGESTED CITATION:',10,citation_tex],'FontSize',8,'LineStyle','none','VerticalAlignment','bottom','FitBoxtoText','off');
export_pdf( ['minipacket_temp/4c- ', engine.name,' - Deac Disabled - BSFC.pdf'])
export_emf( ['minipacket_temp/4d- ', engine.name,' - Deac Disabled - BSFC.emf'])

% BTE Plot
REVS_plot_engine( engine, 'efficiency', 'show_bmep','no_min_bsfc','no_na_curve' , 'show_redline','deac_prop','Disabled');
set(gca,'Position',[ 0.13, 0.17, 0.775, 0.74]);
annotation('textbox',[0.015,0.0,0.98,0.09],'string',['SUGGESTED CITATION:',10,citation_tex],'FontSize',8,'LineStyle','none','VerticalAlignment','bottom','FitBoxtoText','off');
export_pdf( ['minipacket_temp/5c- ', engine.name,' - Deac Disabled - Efficiency.pdf'])
export_emf( ['minipacket_temp/5d- ', engine.name,' - Deac Disabled - Efficiency.emf'])

else
	
% BSFC Plot
REVS_plot_engine( engine, 'BSFC', 'show_bmep','no_min_bsfc','no_na_curve', 'show_redline' );
set(gca,'Position',[ 0.13, 0.17, 0.775, 0.74]);
annotation('textbox',[0.015,0.0,0.98,0.09],'string',['SUGGESTED CITATION:',10,citation_tex],'FontSize',8,'LineStyle','none','VerticalAlignment','bottom','FitBoxtoText','off');
export_pdf( ['minipacket_temp/4a- ', engine.name,' - BSFC.pdf'])
export_emf( ['minipacket_temp/4b- ', engine.name,' - BSFC.emf'])

% BTE Plot
REVS_plot_engine( engine, 'efficiency', 'show_bmep','no_min_bsfc','no_na_curve' , 'show_redline');
set(gca,'Position',[ 0.13, 0.17, 0.775, 0.74]);
annotation('textbox',[0.015,0.0,0.98,0.09],'string',['SUGGESTED CITATION:',10,citation_tex],'FontSize',8,'LineStyle','none','VerticalAlignment','bottom','FitBoxtoText','off');
export_pdf( ['minipacket_temp/5a- ', engine.name,' - Efficiency.pdf'])
export_emf( ['minipacket_temp/5b- ', engine.name,' - Efficiency.emf'])

end

% spreadsheet containing the maps
export_fuelmap_spreadsheet( engine , ['minipacket_temp/6- ', engine.name,' - Fuel Map Data.xlsx'], citation_raw )

cd minipacket_temp
zip( out_zip_file,'*.*')
cd ..

rmdir('minipacket_temp','s')

end

function export_pdf( file )

fig = gcf;

set(fig,'PaperSize', [8.5,11], 'PaperOrientation','landscape','PaperPosition', [0 0 11 8.5]);

style = hgexport('factorystyle');
style.Format = 'pdf';
style.PSLevel = 2;
style.Width = 11.0;
style.Height = 8.5;
style.ScaledFontSize = 150;

hgexport(fig,file,style)

end

function export_emf( file )

fig = gcf;

set(fig,'color','none','Inverthardcopy','off');

style = hgexport('factorystyle');

style.Format = 'meta';
style.Width = 6.5  ;
style.Height = 4.9;
style.ScaledFontSize = 100;

hgexport(fig,file,style)

end

function export_fuelmap_spreadsheet( engine , file, citation)

[speed_radps, torque_Nm] = meshgrid( engine.fuel_map_speed_radps, engine.fuel_map_torque_Nm);
max_torque_Nm = interp1( engine.full_throttle_speed_radps, engine.full_throttle_torque_Nm, speed_radps );

fuel_gps = engine.fuel_map_gps;
fuel_bsfc_gpkWh = engine.fuel_map_gps ./ ( speed_radps .* torque_Nm ) * 1000 * 3600 ;
fuel_eff_pct = 100 * ( speed_radps .* torque_Nm / 1000) ./ ( engine.fuel_map_gps * engine.fuel.energy_density_MJpkg );

keep = torque_Nm < max_torque_Nm & torque_Nm >= 0 & speed_radps > min(engine.idle_speed_radps.table)*0.9 & speed_radps < REVS_calc_engine_power_speed(engine, 0.95, 'max');

sheet = 'Fuel Map';

xls = xlseditor( file);

xls.add_sheet(sheet);
xls.delete_sheet('Sheet1');

xls.write({[engine.name,' - Fuel Map Data'];'SUGGESTED CITATION:';citation;''},'A1:A3');

% xlswrite(file,{[engine.name,' Fuel Map'];'SUGGESTED CITATION:';citation;''},sheet,'A1')

if engine.has_deac
	
	deac_fuel_gps = engine.deac_fuel_map_gps;
	deac_bsfc_gpkWh = engine.deac_fuel_map_gps ./ ( speed_radps .* torque_Nm ) * 1000 * 3600 ;
	deac_eff_pct = 100 * ( speed_radps .* torque_Nm / 1000) ./ ( engine.deac_fuel_map_gps * engine.fuel.energy_density_MJpkg );
	
	header =	{'',		'',			'Cylinder Deac Inactive','','',				'Cylinder Deac Active','','';
				'Speed',	'Torque',	'Fuel Rate',	'BSFC',		'Efficiency',	'Fuel Rate',	'BSFC',		'Efficiency';
				'RPM',		'Nm',		'g/sec',		'g/kW*hr',	'%',			'g/sec',		'g/kW*hr',	'%'	};
	
	xls.write(  header,'A5')
	
	data = [ speed_radps(:)* unit_convert.radps2rpm, torque_Nm(:), fuel_gps(:), fuel_bsfc_gpkWh(:), fuel_eff_pct(:), deac_fuel_gps(:), deac_bsfc_gpkWh(:), deac_eff_pct(:)];
	data(~keep,:) = [];
	
	
	xls.write( data, 'A8')
	xls.format('A:A','NumberFormat','0','ColWidth',12);
	xls.format( 'B:B','NumberFormat','0.00','ColWidth',12);
	xls.format( 'D:E','NumberFormat','0.0','ColWidth',12);
	xls.format( 'G:H','NumberFormat','0.0','ColWidth',12);
	
	xls.format( 'C:C','NumberFormat','0.0000','ColWidth',12);
	xls.format('F:F','NumberFormat','0.0000','ColWidth',12);
	
	xls.format('A1:H1','MergeCells',true)
	xls.format('A2:H2','MergeCells',true)
	xls.format('A3:H3','MergeCells',true)
	xls.format('C5:E5','MergeCells',true)
	xls.format('F5:H5','MergeCells',true)
	
	
	xls.format('A1','HorizAlign','center','FontBold',true);
	xls.format('A5:H7','HorizAlign','center','FontBold',true);
	
	xls.format( '3:3','WrapText',true,'RowHeight',30);
	
else
	
	header =	{	'Speed',	'Torque',	'Fuel Rate',	'BSFC',		'Efficiency'	;
					'RPM',		'Nm',		'g/sec',		'g/kW*hr',	'%'			};
	
	xls.write(header, 'A5')
	
	data = [ speed_radps(:)* unit_convert.radps2rpm, torque_Nm(:), fuel_gps(:), fuel_bsfc_gpkWh(:), fuel_eff_pct(:)];
	data(~keep,:) = [];
	
	xls.write( data,'A7')
	xls.format( 'A:A','NumberFormat','0','ColWidth',19);
	xls.format( 'B:B','NumberFormat','0.00','ColWidth',19);
	xls.format( 'D:E','NumberFormat','0.0','ColWidth',19);
	xls.format( 'C:C','NumberFormat','0.0000','ColWidth',19);
	
	xls.format('A1:E1','MergeCells',true)
	xls.format('A2:E2','MergeCells',true)
	xls.format('A3:E3','MergeCells',true)
	
	
	xls.format('A1','HorizAlign','center','FontBold',true);	
	xls.format('A5:H6','HorizAlign','center','FontBold',true);
	
	xls.format('3:3','WrapText',true,'RowHeight',30);
	
end

xls.close(1);

end



